package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// rejectInvitation updates an invitation to status REJECTED with a reason specified in parameter.
func (d *datastore) rejectInvitation(ctx context.Context, invitationID string, reasonRejected models.InvitationReasonRejected) (*models.DBInvitation, error) {
	query := `
		UPDATE squad_invitations
		SET 
		  status = $1,
		  reason_rejected = $2,
		  updated_at = $3
		WHERE id = $4
		RETURNING
		  id, squad_id, sender_id, recipient_id, status, reason_rejected, created_at, updated_at`

	row := d.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		query,
		models.NewDBFromInvitationStatus(models.InvitationStatusRejected),
		models.NewDBFromInvitationReasonRejected(reasonRejected),
		d.now(),
		invitationID,
	)

	invitation := models.DBInvitation{}
	err := row.Scan(
		&invitation.ID,
		&invitation.SquadID,
		&invitation.SenderID,
		&invitation.RecipientID,
		&invitation.Status,
		&invitation.ReasonRejected,
		&invitation.CreatedAt,
		&invitation.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not reject invitation for invitationID %v", invitationID)
	}

	return &invitation, nil
}

// RejectInvitation records stats for rejectInvitation.
func (d *datastore) RejectInvitation(ctx context.Context, invitationID string, reasonRejected models.InvitationReasonRejected) (*models.DBInvitation, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "reject_invitation", startTime, err == nil)
	}(time.Now())

	invitation, err := d.rejectInvitation(ctx, invitationID, reasonRejected)
	return invitation, err
}
