package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// updateInvitationStatus updates the squad's status and returns the db representation of the updated invitation.
func (d *datastore) updateInvitationStatus(ctx context.Context, invitationID string, status models.InvitationStatus) (*models.DBInvitation, error) {
	query := `
		UPDATE squad_invitations
		SET 
		  status = $1,
		  updated_at = $2
		WHERE id = $3
		RETURNING
		  id, squad_id, sender_id, recipient_id, status, reason_rejected, created_at, updated_at`

	row := d.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		query,
		models.NewDBFromInvitationStatus(status),
		d.now(),
		invitationID,
	)

	invitation := models.DBInvitation{}
	err := row.Scan(
		&invitation.ID,
		&invitation.SquadID,
		&invitation.SenderID,
		&invitation.RecipientID,
		&invitation.Status,
		&invitation.ReasonRejected,
		&invitation.CreatedAt,
		&invitation.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not update invitation status to %v for ID %v", status, invitationID)
	}

	return &invitation, nil
}

// UpdateInvitationStatus records stats for updateInvitationStatus.
func (d *datastore) UpdateInvitationStatus(ctx context.Context, invitationID string, status models.InvitationStatus) (*models.DBInvitation, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "update_invitation_status", startTime, err == nil)
	}(time.Now())

	invitation, err := d.updateInvitationStatus(ctx, invitationID, status)
	return invitation, err
}
