package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// updateInvitePolicyByChannelID updates the channel's invite policy and returns the db representation of the updated policy.
func (d *datastore) updateInvitePolicyByChannelID(ctx context.Context, channelID, callerID string, invitePolicy models.InvitationPolicy) (*models.DBInvitePolicy, error) {
	query := `
		INSERT INTO squad_invite_policy (
			channel_id,
			caller_id,
			invite_policy,
			created_at,
			updated_at
		) VALUES (
			$1, $2, $3, $4, $4
		) ON CONFLICT (
			channel_id
		)
		DO UPDATE SET
			invite_policy = $3,
			updated_at = $4
		RETURNING
		  channel_id, caller_id, invite_policy, created_at, updated_at`

	row := d.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		query,
		channelID,
		callerID,
		models.NewDBFromInvitationPolicy(invitePolicy),
		d.now(),
	)

	updatedInvitePolicy := models.DBInvitePolicy{}
	err := row.Scan(
		&updatedInvitePolicy.ChannelID,
		&updatedInvitePolicy.CallerID,
		&updatedInvitePolicy.InvitePolicy,
		&updatedInvitePolicy.CreatedAt,
		&updatedInvitePolicy.UpdatedAt,
	)
	if err == sql.ErrNoRows {
		return nil, nil
	} else if err != nil {
		return nil, errors.Wrapf(err, "could not update invite policy to %v for ID %v", invitePolicy, channelID)
	}

	return &updatedInvitePolicy, nil
}

// UpdateInvitePolicyByChannelID records stats for updateInvitePolicyByChannelID.
func (d *datastore) UpdateInvitePolicyByChannelID(ctx context.Context, channelID, callerID string, invitePolicy models.InvitationPolicy) (*models.DBInvitePolicy, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "update_invite_policy_by_channel_id", startTime, err == nil)
	}(time.Now())

	updatedInvitePolicy, err := d.updateInvitePolicyByChannelID(ctx, channelID, callerID, invitePolicy)
	return updatedInvitePolicy, err
}
