package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// updateMemberStatusByID updates the membership's status and returns the db representation of the updated member.
func (d *datastore) updateMemberStatusByID(ctx context.Context, id string, status models.MemberStatus) (*models.DBMember, error) {
	var row *sql.Row

	if status == models.MemberStatusDeleted {
		query := `
			UPDATE squad_members
			SET 
			  status = $1
			  deleted_at = $2
			WHERE id = $3
			RETURNING
			  id, squad_id, member_id, created_at, deleted_at, status`
		row = d.getTxIfJoined(ctx).QueryRowContext(
			ctx,
			query,
			models.NewDBFromMemberStatus(status),
			d.now(),
			id,
		)
	} else {
		query := `
			UPDATE squad_members
			SET 
			  status = $1
			WHERE id = $2
			RETURNING
			  id, squad_id, member_id, created_at, deleted_at, status`

		row = d.getTxIfJoined(ctx).QueryRowContext(
			ctx,
			query,
			models.NewDBFromMemberStatus(status),
			id,
		)
	}

	member := models.DBMember{}
	err := row.Scan(
		&member.ID,
		&member.SquadID,
		&member.MemberID,
		&member.CreatedAt,
		&member.DeletedAt,
		&member.Status,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not update member status to %v for ID %v", status, id)
	}

	return &member, nil
}

// UpdateMemberStatusByID records stats for updateMemberStatusByID.
func (d *datastore) UpdateMemberStatusByID(ctx context.Context, id string, status models.MemberStatus) (*models.DBMember, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "update_member_status_by_id", startTime, err == nil)
	}(time.Now())

	member, err := d.updateMemberStatusByID(ctx, id, status)
	return member, err
}
