package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// updateSquadOwner updates the squad's owner and returns the db representation of the updated squad.
func (d *datastore) updateSquadOwner(ctx context.Context, squadID string, ownerID string) (*models.DBSquad, error) {
	query := `
		UPDATE squads
		SET 
		  owner_id = $1,
		  updated_at = $2
		WHERE id = $3
		RETURNING
		  id, owner_id, status, created_by, created_at, updated_at`

	row := d.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		query,
		ownerID,
		d.now(),
		squadID,
	)

	squad := models.DBSquad{}
	err := row.Scan(
		&squad.ID,
		&squad.OwnerID,
		&squad.Status,
		&squad.CreatedBy,
		&squad.CreatedAt,
		&squad.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not update squad owner to %v for squad ID %v", ownerID, squadID)
	}

	return &squad, nil
}

// UpdateSquadOwner records stats for updateSquadOwner.
func (d *datastore) UpdateSquadOwner(ctx context.Context, squadID string, ownerID string) (*models.DBSquad, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "update_squad_owner", startTime, err == nil)
	}(time.Now())

	squad, err := d.updateSquadOwner(ctx, squadID, ownerID)
	return squad, err
}
