package datastore

import (
	"context"
	"database/sql"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// updateSquadStatus updates the squad's status and returns the db representation of the updated squad.
func (d *datastore) updateSquadStatus(ctx context.Context, squadID string, status models.SquadStatus) (*models.DBSquad, error) {
	query := `
		UPDATE squads
		SET 
		  status = $1,
		  updated_at = $2
		WHERE id = $3
		RETURNING
		  id, owner_id, status, created_by, created_at, updated_at`

	row := d.getTxIfJoined(ctx).QueryRowContext(
		ctx,
		query,
		models.NewDBFromSquadStatus(status),
		d.now(),
		squadID,
	)

	squad := models.DBSquad{}
	err := row.Scan(
		&squad.ID,
		&squad.OwnerID,
		&squad.Status,
		&squad.CreatedBy,
		&squad.CreatedAt,
		&squad.UpdatedAt,
	)
	switch {
	case err == sql.ErrNoRows:
		return nil, nil
	case err != nil:
		return nil, errors.Wrapf(err, "could not update squad status to %v for ID %v", status, squadID)
	}

	return &squad, nil
}

// UpdateSquad records stats for updateSquad.
func (d *datastore) UpdateSquadStatus(ctx context.Context, squadID string, status models.SquadStatus) (*models.DBSquad, error) {
	var err error
	defer func(startTime time.Time) {
		d.recordStats(ctx, "update_squad_status", startTime, err == nil)
	}(time.Now())

	squad, err := d.updateSquadStatus(ctx, squadID, status)
	return squad, err
}
