package models

import (
	"database/sql"
	"strings"

	"code.justin.tv/twitch-events/meepo/rpc/meepo"

	"google.golang.org/protobuf/types/known/timestamppb"
)

// NewProtoFromSquad converts a Squad into a proto Squad.
func NewProtoFromSquad(s *Squad) (*meepo.Squad, error) {
	if s == nil {
		return nil, nil
	}

	createdAt := timestamppb.New(s.CreatedAt)

	updatedAt := timestamppb.New(s.UpdatedAt)

	r := &meepo.Squad{
		Id:        s.ID,
		MemberIds: s.MemberIDs,
		Status:    NewProtoFromSquadStatus(s.Status),
		CreatedAt: createdAt,
		UpdatedAt: updatedAt,
	}

	if s.OwnerID != nil {
		r.OwnerId = *s.OwnerID
	}

	return r, nil
}

// NewSquadFromManagedSquad converts a ManagedSquad to a Squad.
func NewSquadFromManagedSquad(s *ManagedSquad) *Squad {
	if s == nil {
		return nil
	}
	return &Squad{
		ID:        s.ID,
		MemberIDs: s.MemberIDs,
		OwnerID:   s.OwnerID,
		Status:    s.Status,
		CreatedAt: s.CreatedAt,
		CreatedBy: s.CreatedBy,
		UpdatedAt: s.UpdatedAt,
	}
}

// NewManagedSquadFromDB creates a ManagedSquad from a set of DB objects.
func NewManagedSquadFromDB(s *DBSquad, members []*DBMember, invitations []*DBInvitation) *ManagedSquad {
	if s == nil {
		return nil
	}

	r := &ManagedSquad{
		ID:        s.ID,
		Status:    NewSquadStatusFromDB(s.Status),
		CreatedBy: s.CreatedBy,

		CreatedAt: s.CreatedAt,
		UpdatedAt: s.UpdatedAt,

		Invitations: NewSquadInvitationsFromDB(invitations),
	}

	if s.OwnerID.Valid {
		r.OwnerID = &s.OwnerID.String
	}

	if members != nil {
		memberIDs := make([]string, 0, len(members))
		for _, member := range members {
			if member == nil {
				continue
			}
			if member.SquadID == r.ID && !member.DeletedAt.Valid {
				memberIDs = append(memberIDs, member.MemberID)
			}
		}

		r.MemberIDs = memberIDs
	}

	return r
}

// NewSquadFromDB creates a Squad from a set of DB objects.
func NewSquadFromDB(s *DBSquad, members []*DBMember) *Squad {
	if s == nil {
		return nil
	}

	r := &Squad{
		ID:        s.ID,
		Status:    NewSquadStatusFromDB(s.Status),
		CreatedBy: s.CreatedBy,

		CreatedAt: s.CreatedAt,
		UpdatedAt: s.UpdatedAt,
	}

	if s.OwnerID.Valid {
		r.OwnerID = &s.OwnerID.String
	}

	if members == nil {
		return r
	}

	memberIDs := make([]string, 0, len(members))
	for _, member := range members {
		if member == nil {
			continue
		}
		if member.SquadID == r.ID && !member.DeletedAt.Valid {
			memberIDs = append(memberIDs, member.MemberID)
		}
	}

	r.MemberIDs = memberIDs

	return r
}

// NewPubsubManagedSquad creates a PubsubManagedSquad.
func NewPubsubManagedSquad(s *ManagedSquad, m []PubsubUser, i []PubsubOutgoingInvitation) *PubsubManagedSquad {
	if s == nil {
		return nil
	}
	return &PubsubManagedSquad{
		ID:          s.ID,
		Members:     m,
		OwnerID:     s.OwnerID,
		Status:      s.Status,
		Invitations: i,
	}
}

// NewProtoFromInvitation converts an Invitation into a proto Invitation.
func NewProtoFromInvitation(i *Invitation) (*meepo.Invitation, error) {
	if i == nil {
		return nil, nil
	}

	createdAt := timestamppb.New(i.CreatedAt)

	updatedAt := timestamppb.New(i.UpdatedAt)

	r := &meepo.Invitation{
		Id:             i.ID,
		SquadId:        i.SquadID,
		SenderId:       i.SenderID,
		RecipientId:    i.RecipientID,
		Status:         NewProtoFromInvitationStatus(i.Status),
		ReasonRejected: NewProtoFromInvitationReasonRejected(i.ReasonRejected),
		NetworkType:    NewProtoFromInvitationNetworkType(i.NetworkType),
		CreatedAt:      createdAt,
		UpdatedAt:      updatedAt,
	}

	return r, nil
}

// NewPubsubOutgoingInvitation creates a PubsubOutgoingInvitation from a OutgoingInvitation.
func NewPubsubOutgoingInvitation(i *OutgoingInvitation, recipient, sender PubsubUser) *PubsubOutgoingInvitation {
	if i == nil {
		return nil
	}
	return &PubsubOutgoingInvitation{
		ID:             i.ID,
		Sender:         sender,
		Recipient:      recipient,
		Status:         i.Status,
		ReasonRejected: i.ReasonRejected,
		CreatedAt:      i.CreatedAt,
		UpdatedAt:      i.UpdatedAt,
	}
}

// NewInvitationFromDB converts a DBInvitation into an Invitation.
func NewInvitationFromDB(i *DBInvitation) *Invitation {
	if i == nil {
		return nil
	}

	r := &Invitation{
		ID:             i.ID,
		SquadID:        i.SquadID,
		SenderID:       i.SenderID,
		RecipientID:    i.RecipientID,
		Status:         NewInvitationStatusFromDB(i.Status),
		ReasonRejected: NewInvitationReasonRejectedFromDB(i.ReasonRejected),

		CreatedAt: i.CreatedAt,
		UpdatedAt: i.UpdatedAt,
	}
	return r
}

// NewSquadInvitationFromDB converts a DBInvitation into an OutgoingInvitation.
func NewSquadInvitationFromDB(i *DBInvitation) *OutgoingInvitation {
	if i == nil {
		return nil
	}

	return &OutgoingInvitation{
		ID:             i.ID,
		SquadID:        i.SquadID,
		SenderID:       i.SenderID,
		RecipientID:    i.RecipientID,
		Status:         NewInvitationStatusFromDB(i.Status),
		ReasonRejected: NewInvitationReasonRejectedFromDB(i.ReasonRejected),
		CreatedAt:      i.CreatedAt,
		UpdatedAt:      i.UpdatedAt,
	}
}

// NewSquadInvitationsFromDB converts a slice of DBInvitations into a slice of Invitations.
func NewSquadInvitationsFromDB(dbInvitations []*DBInvitation) []*OutgoingInvitation {
	r := make([]*OutgoingInvitation, len(dbInvitations))
	for i, dbInvitation := range dbInvitations {
		r[i] = NewSquadInvitationFromDB(dbInvitation)
	}
	return r
}

// NewMemberFromDB converts a DBMember into a Member.
func NewMemberFromDB(m *DBMember) *Member {
	if m == nil {
		return nil
	}

	r := &Member{
		ID:       m.ID,
		SquadID:  m.SquadID,
		MemberID: m.MemberID,
		Status:   NewMemberStatusFromDB(m.Status),

		CreatedAt: m.CreatedAt,
	}

	if m.DeletedAt.Valid {
		r.DeletedAt = &m.DeletedAt.Time
	}
	return r
}

// NewMemberFromDBMemberAndSquad converts a DBMemberAndSquad into a Member.
func NewMemberFromDBMemberAndSquad(m *DBMemberAndSquad) *Member {
	if m == nil {
		return nil
	}

	r := &Member{
		ID:       m.Member.ID,
		SquadID:  m.Member.SquadID,
		MemberID: m.Member.MemberID,
		Status:   NewMemberStatusFromDB(m.Member.Status),

		CreatedAt: m.Member.CreatedAt,
	}

	if m.Member.DeletedAt.Valid {
		r.DeletedAt = &m.Member.DeletedAt.Time
	}
	return r
}

// NewProtoFromMember converts a Member into a proto Member.
func NewProtoFromMember(m *Member) (*meepo.Member, error) {
	if m == nil {
		return nil, nil
	}

	createdAt := timestamppb.New(m.CreatedAt)

	r := &meepo.Member{
		Id:        m.ID,
		SquadId:   m.SquadID,
		MemberId:  m.MemberID,
		Status:    NewProtoFromMemberStatus(m.Status),
		CreatedAt: createdAt,
	}

	if m.DeletedAt != nil {
		deletedAt := timestamppb.New(*m.DeletedAt)

		r.DeletedAt = deletedAt
	}
	return r, nil
}

// NewDBFromSquadStatus converts a SquadStatus into its DB representation.
func NewDBFromSquadStatus(s SquadStatus) string {
	return strings.ToLower(string(s))
}

// NewSquadStatusFromDB converts a DB squad status into a SquadStatus.
func NewSquadStatusFromDB(s string) SquadStatus {
	return SquadStatus(strings.ToUpper(s))
}

// NewProtoFromSquadStatus converts a SquadStatus into a proto Squad_Status.
func NewProtoFromSquadStatus(s SquadStatus) meepo.Squad_Status {
	status, ok := meepo.Squad_Status_value[string(s)]
	if !ok {
		return meepo.Squad_UNKNOWN
	}
	return meepo.Squad_Status(status)
}

// NewSquadStatusFromProto converts a proto Squad_Status into a SquadStatus.
func NewSquadStatusFromProto(s meepo.Squad_Status) SquadStatus {
	return SquadStatus(meepo.Squad_Status_name[int32(s)])
}

// NewDBFromInvitationStatus converts a InvitationStatus into its DB representation.
func NewDBFromInvitationStatus(s InvitationStatus) string {
	return strings.ToLower(string(s))
}

// NewProtoFromInvitationStatus converts a InvitationStatus into a proto Invitation_Status.
func NewProtoFromInvitationStatus(s InvitationStatus) meepo.Invitation_Status {
	status, ok := meepo.Invitation_Status_value[string(s)]
	if !ok {
		return meepo.Invitation_UNKNOWN
	}
	return meepo.Invitation_Status(status)
}

// NewInvitationStatusFromDB converts a DBInvitationStatus into an InvitationStatus.
func NewInvitationStatusFromDB(s string) InvitationStatus {
	return InvitationStatus(strings.ToUpper(s))
}

// NewInvitationStatusFromProto converts a proto Invitation_Status into an InvitationStatus.
func NewInvitationStatusFromProto(s meepo.Invitation_Status) InvitationStatus {
	return InvitationStatus(meepo.Invitation_Status_name[int32(s)])
}

// NewProtoFromInvitationNetworkType converts a InvitationNetworkType into a proto Invitation_NetworkType.
func NewProtoFromInvitationNetworkType(t InvitationNetworkType) meepo.Invitation_NetworkType {
	networkType, ok := meepo.Invitation_NetworkType_value[string(t)]
	if !ok {
		return meepo.Invitation_NETWORK_UNKNOWN
	}
	return meepo.Invitation_NetworkType(networkType)
}

// NewProtoFromInvitationReasonRejected converts a InvitationReasonRejected into a proto Invitation_Reason_Rejected.
func NewProtoFromInvitationReasonRejected(s *InvitationReasonRejected) meepo.Invitation_ReasonRejected {
	if s == nil {
		return meepo.Invitation_NOT_REJECTED
	}
	reason, ok := meepo.Invitation_ReasonRejected_value[string(*s)]
	if !ok {
		return meepo.Invitation_NOT_REJECTED
	}
	return meepo.Invitation_ReasonRejected(reason)
}

// NewInvitationReasonRejectedFromDB converts a DBInvitationReasonRejected into an InvitationReasonRejected.
func NewInvitationReasonRejectedFromDB(s sql.NullString) *InvitationReasonRejected {
	if !s.Valid {
		return nil
	}
	reason := InvitationReasonRejected(strings.ToUpper(s.String))
	return &reason
}

// NewDBFromInvitationReasonRejected converts a InvitationReasonRejected into its DB representation.
func NewDBFromInvitationReasonRejected(s InvitationReasonRejected) string {
	return strings.ToLower(string(s))
}

// NewDBFromMemberStatus converts a MemberStatus into its DB representation.
func NewDBFromMemberStatus(s MemberStatus) string {
	return strings.ToLower(string(s))
}

// NewMemberStatusFromDB converts a DBMemberStatus into an MemberStatus.
func NewMemberStatusFromDB(s string) MemberStatus {
	return MemberStatus(strings.ToUpper(s))
}

// NewProtoFromMemberStatus converts a MemberStatus into a proto Member_Status.
func NewProtoFromMemberStatus(s MemberStatus) meepo.Member_Status {
	status, ok := meepo.Member_Status_value[string(s)]
	if !ok {
		return meepo.Member_UNKNOWN
	}
	return meepo.Member_Status(status)
}

// NewInvitePolicyFromDB converts a DB squad invite policy into an InvitePolicy.
func NewInvitePolicyFromDB(p *DBInvitePolicy) *InvitePolicy {
	if p == nil {
		return nil
	}

	r := &InvitePolicy{
		ChannelID:    p.ChannelID,
		CallerID:     p.CallerID,
		InvitePolicy: NewInvitationPolicyFromDB(p.InvitePolicy),
		CreatedAt:    p.CreatedAt,
		UpdatedAt:    p.UpdatedAt,
	}

	return r
}

// NewInvitationPolicyFromDB converts a DBInvitationStatus into an InvitationStatus.
func NewInvitationPolicyFromDB(s string) InvitationPolicy {
	return InvitationPolicy(strings.ToUpper(s))
}

// NewDBFromInvitationPolicy converts an InvitationPolicy into its DB representation.
func NewDBFromInvitationPolicy(p InvitationPolicy) string {
	return strings.ToLower(string(p))
}

// NewProtoFromInvitationPolicy converts an InvitationPolicy into a proto InvitePolicy_InvitePolicy.
func NewProtoFromInvitationPolicy(p InvitationPolicy) meepo.InvitePolicy_InvitePolicy {
	invitePolicy, ok := meepo.InvitePolicy_InvitePolicy_value[string(p)]
	if !ok {
		return meepo.InvitePolicy_NETWORK
	}
	return meepo.InvitePolicy_InvitePolicy(invitePolicy)
}

// NewInvitationPolicyFromProto converts a proto InvitePolicy_InvitePolicy into an InvitationPolicy.
func NewInvitationPolicyFromProto(p meepo.InvitePolicy_InvitePolicy) InvitationPolicy {
	return InvitationPolicy(meepo.InvitePolicy_InvitePolicy_name[int32(p)])
}

// NewSquadIDsFromDBInvitations gets squad IDs from DBInvitations
func NewSquadIDsFromDBInvitations(invs []*DBInvitation) []string {
	var squadIDs []string
	for _, inv := range invs {
		if inv == nil {
			continue
		}
		squadIDs = append(squadIDs, inv.SquadID)
	}

	return squadIDs
}
