package models

import (
	"database/sql"
	"time"
)

// InvitationStatus is an enum of different invitation statuses.
type InvitationStatus string

// InvitationNetworkType is an enum of different invitation network types.
type InvitationNetworkType string

// InvitationReasonRejected is an enum of different invitation rejection reasons.
type InvitationReasonRejected string

// Enum values of invitation statuses.
const (
	InvitationStatusPending  InvitationStatus = "PENDING"
	InvitationStatusAccepted InvitationStatus = "ACCEPTED"
	InvitationStatusRejected InvitationStatus = "REJECTED"
	InvitationStatusDeleted  InvitationStatus = "DELETED"
)

// Enum values of invitation network types.
const (
	InvitationInNetwork    InvitationNetworkType = "IN_NETWORK"
	InvitationOutOfNetwork InvitationNetworkType = "OUT_OF_NETWORK"
)

// Enum values of invitation rejection reasons.
const (
	InvitationRecipientRejected        InvitationReasonRejected = "RECIPIENT_REJECTED"
	InvitationRecipientMaxLimitReached InvitationReasonRejected = "RECIPIENT_MAX_LIMIT_REACHED"
)

// Invitation represents a Twitch squad invitation.
type Invitation struct {
	ID             string
	SquadID        string
	SenderID       string
	RecipientID    string
	Status         InvitationStatus
	NetworkType    InvitationNetworkType
	ReasonRejected *InvitationReasonRejected

	CreatedAt time.Time
	UpdatedAt time.Time
}

// OutgoingInvitation contains the info a sender needs about an invitation they created.
type OutgoingInvitation struct {
	ID             string
	SquadID        string
	SenderID       string
	RecipientID    string
	Status         InvitationStatus
	ReasonRejected *InvitationReasonRejected

	CreatedAt time.Time
	UpdatedAt time.Time
}

// InvitationResponse wraps a cursor with an invitation.
type InvitationResponse struct {
	Invitation *Invitation
	Cursor     string
}

// DBInvitation represents a squad invitation in Meepo DB.
type DBInvitation struct {
	ID             string
	SquadID        string
	SenderID       string
	RecipientID    string
	Status         string
	ReasonRejected sql.NullString

	CreatedAt time.Time
	UpdatedAt time.Time
}

// PubsubOutgoingInvitation represents a squad's outgoing invitations.
type PubsubOutgoingInvitation struct {
	ID             string                    `json:"id"`
	Sender         PubsubUser                `json:"sender"`
	Recipient      PubsubUser                `json:"recipient"`
	Status         InvitationStatus          `json:"status"`
	ReasonRejected *InvitationReasonRejected `json:"reason_rejected"`

	CreatedAt time.Time `json:"created_at"`
	UpdatedAt time.Time `json:"updated_at"`
}

// PubsubIncomingInvitation represents a squad invitation that a user has received.
type PubsubIncomingInvitation struct {
	ID             string                    `json:"id"`
	SquadID        string                    `json:"squad_id"`
	Sender         PubsubUser                `json:"sender"`
	Recipient      PubsubUser                `json:"recipient"`
	Status         InvitationStatus          `json:"status"`
	NetworkType    InvitationNetworkType     `json:"network_type"`
	ReasonRejected *InvitationReasonRejected `json:"reason_rejected"`

	CreatedAt time.Time `json:"created_at"`
	UpdatedAt time.Time `json:"updated_at"`
}

// CreatePendingInvitationInput contains the possible input parameters that go into a CreateInvitation operation.
type CreatePendingInvitationInput struct {
	SquadID     string
	SenderID    string
	RecipientID string
}
