package models

import (
	"database/sql"
	"time"
)

// SquadStatus is an enum of different squad statuses.
type SquadStatus string

// Enum values of squad statuses.
const (
	SquadStatusPending SquadStatus = "PENDING"
	SquadStatusLive    SquadStatus = "LIVE"
	SquadStatusEnded   SquadStatus = "ENDED"
)

// Squad represents a Twitch squad of up to 4 channels.
// Remember to update the cache representation if you need to update this, and update the cache key if the change is
// backwards incompatible with previously cached values.
type Squad struct {
	ID        string      `json:"id"`
	MemberIDs []string    `json:"member_ids"`
	OwnerID   *string     `json:"owner_id"`
	Status    SquadStatus `json:"status"`
	CreatedBy string      `json:"created_by"`

	CreatedAt time.Time `json:"created_at"`
	UpdatedAt time.Time `json:"updated_at"`
}

// ManagedSquad contains the information about a squad that an owner is allowed to see to manage the squad.
type ManagedSquad struct {
	ID        string
	MemberIDs []string
	OwnerID   *string
	Status    SquadStatus
	CreatedBy string

	CreatedAt time.Time
	UpdatedAt time.Time

	Invitations []*OutgoingInvitation
}

// DBSquad represents a squad in Meepo DB.
type DBSquad struct {
	ID        string
	OwnerID   sql.NullString
	Status    string
	CreatedBy string

	CreatedAt time.Time
	UpdatedAt time.Time
}

// PubsubSquad represents a Twitch squad used for Pubsub messages.
type PubsubSquad struct {
	ID      string       `json:"id"`
	Members []PubsubUser `json:"members"`
	OwnerID *string      `json:"owner_id"`
	Status  SquadStatus  `json:"status"`
}

// PubsubManagedSquad is the info needed to manage a squad that gets sent to Pubsub.
type PubsubManagedSquad struct {
	ID          string                     `json:"id"`
	OwnerID     *string                    `json:"owner_id"`
	Status      SquadStatus                `json:"status"`
	Members     []PubsubUser               `json:"members"`
	Invitations []PubsubOutgoingInvitation `json:"invitations"`
}

// CreateSquadInput contains the possible input parameters that go into a CreateSquad operation.
type CreateSquadInput struct {
	OwnerID   string
	CreatedBy string
	Status    SquadStatus
}
