package models

import "time"

// Possible tracking event table names.
const (
	CreatorActionTrackingEvent    string = "squad_stream_creator_action"
	SquadStateChangeTrackingEvent string = "squad_stream_state_change"
	SquadStreamErrorTrackingEvent string = "squad_stream_error"
)

// CreatorActionTracking is a tracking event that will be pulled into the squad_stream_creator_action table.
type CreatorActionTracking struct {
	ChannelID     int               `json:"channel_id"`
	Channel       string            `json:"channel"`
	NumMembers    int               `json:"num_members"`
	SquadStreamID string            `json:"squad_stream_id"`
	IsLive        bool              `json:"is_live"` // Is the channel live?
	IsOwner       bool              `json:"is_owner"`
	SquadViewable bool              `json:"squad_viewable"` // Is the squad live?
	InviteID      string            `json:"invite_id,omitempty"`
	Action        CreatorActionType `json:"action"`
	Method        CreatorMethodType `json:"method,omitempty"`
}

// CreatorActionType is an enum for managing the actions we are tracking in the squad_stream_creator_action table.
type CreatorActionType string

// Enum values for CreatorActionType.
const (
	CreatorActionTypeAcceptInvite CreatorActionType = "accept_invite"
	CreatorActionTypeCancelInvite CreatorActionType = "cancel_invite"
	CreatorActionTypeRejectInvite CreatorActionType = "reject_invite"
	CreatorActionTypeSendInvite   CreatorActionType = "send_invite"

	// In the tracking spec, "join_squad" is defined as becoming a part of a
	// squad that has started. All squad members who have accepted invitations
	// to a pending squad will "join" that squad when it is started.
	CreatorActionTypeJoinSquad CreatorActionType = "join_squad"

	CreatorActionTypeLeaveSquad      CreatorActionType = "leave_squad"
	CreatorActionTypeStartSquad      CreatorActionType = "start_squad"
	CreatorActionTypeRemoveFromSquad CreatorActionType = "remove_from_squad"

	CreatorActionTypeInheritOwnership CreatorActionType = "inherit_ownership"
)

// CreatorMethodType is an enum for managing the methods we are tracking in the squad_stream_creator_action table.
type CreatorMethodType string

// Enum values for CreatorMethodType.
const (
	CreatorMethodTypeClickAccept     CreatorMethodType = "click_accept"
	CreatorMethodTypeClickLeaveSquad CreatorMethodType = "click_leave_squad"

	CreatorMethodTypeOwnerClickCancel   CreatorMethodType = "owner_click_cancel"
	CreatorMethodTypeOwnerClickStart    CreatorMethodType = "owner_click_start"
	CreatorMethodTypeOwnerClickRemove   CreatorMethodType = "owner_click_remove"
	CreatorMethodTypeOriginalOwnerLeave CreatorMethodType = "original_owner_leave"

	CreatorMethodTypeGoOffline CreatorMethodType = "go_offline"
)

// CreatorActionTrackingEventSet represents a group of creator_action_tracking_events to fire.
type CreatorActionTrackingEventSet struct {
	SquadID      string
	InvitationID string

	MemberIDs   []string
	OwnerID     *string
	SquadStatus *SquadStatus

	Events []CreatorActionTrackingEventInfo
}

// CreatorActionTrackingEventInfo represents the information needed to fire each individual tracking event in a set.
type CreatorActionTrackingEventInfo struct {
	ChannelID     string
	DisplayName   *string
	ChannelIsLive *bool
	CreatorAction CreatorActionType
	CreatorMethod CreatorMethodType
}

// SquadStateChangeTracking is a tracking event that will be pulled into the squad_stream_state_change table.
type SquadStateChangeTracking struct {
	SquadStreamID  string                `json:"squad_stream_id"`
	State          SquadStatus           `json:"state"`
	PreviousState  SquadStatus           `json:"previous_state,omitempty"`
	OwnerChannelID int                   `json:"owner_channel_id"`
	OwnerChannel   string                `json:"owner_channel"`
	LastUpdated    *time.Time            `json:"last_updated,omitempty"`
	NumMembers     int                   `json:"num_members"`
	NumMembersLive int                   `json:"num_members_live"`
	Method         StateChangeMethodType `json:"method"`
}

// StateChangeMethodType is an enum for managing the methods we are tracking in the squad_stream_creator_action table.
type StateChangeMethodType string

// Enum values for StateChangeMethodType.
const (
	StateChangeMethodTypeFirstInviteSent   StateChangeMethodType = "first_invite_sent"
	StateChangeMethodTypeOwnerStartSquad   StateChangeMethodType = "owner_start_squad"
	StateChangeMethodTypeLastMemberLeave   StateChangeMethodType = "last_member_leave"
	StateChangeMethodTypeLastMemberOffline StateChangeMethodType = "last_member_offline"
)

// SquadStateChangeTrackingEventInfo represents the information needed to fire a state change tracking event.
type SquadStateChangeTrackingEventInfo struct {
	SquadID       string
	State         SquadStatus
	PreviousState *SquadStatus
	Method        StateChangeMethodType
	LastUpdated   *time.Time
	NumMembers    *int
}

// SquadStreamErrorTracking is a tracking event that will be pulled into the squad_stream_error table.
type SquadStreamErrorTracking struct {
	ChannelID       int             `json:"channel_id"`
	Channel         string          `json:"channel"`
	TargetChannelID int             `json:"target_channel_id"`
	TargetChannel   string          `json:"target_channel"`
	NumMembers      int             `json:"num_members"`
	SquadStreamID   string          `json:"squad_stream_id"`
	InviteID        string          `json:"invite_id,omitempty"`
	ErrorCode       string          `json:"error_code"`
	Method          ErrorMethodType `json:"method,omitempty"`
}

// ErrorMethodType is an enum for managing the methods we are tracking in the squad_stream_error table.
type ErrorMethodType string

// Enum values for ErrorMethodType.
const (
	ErrorMethodTypeAcceptInvitation                        ErrorMethodType = "accept_invitation"
	ErrorMethodTypeCreateInvitation                        ErrorMethodType = "create_invitation"
	ErrorMethodTypeCreateMembership                        ErrorMethodType = "create_membership"
	ErrorMethodTypeCreateSquad                             ErrorMethodType = "create_squad"
	ErrorMethodTypeDeleteInvitation                        ErrorMethodType = "delete_invitation"
	ErrorMethodTypeDeleteMember                            ErrorMethodType = "delete_member"
	ErrorMethodTypeGetInvitationsBySquadID                 ErrorMethodType = "get_invitations_by_squad_id"
	ErrorMethodTypeGetInvitePolicyByChannelID              ErrorMethodType = "get_invite_policy_by_channel_id"
	ErrorMethodTypeGetPendingInvitationsByRecipientID      ErrorMethodType = "get_pending_invitations_by_recipient_id"
	ErrorMethodTypeGetPendingInvitationsCountByRecipientID ErrorMethodType = "get_pending_invitations_count_by_recipient_id"
	ErrorMethodTypeLeaveSquad                              ErrorMethodType = "leave_squad"
	ErrorMethodTypePrepareToCreateInvitation               ErrorMethodType = "prepare_to_create_invitation"
	ErrorMethodTypeRejectInvitation                        ErrorMethodType = "reject_invitation"
	ErrorMethodTypeRejectOutOfNetworkInvitations           ErrorMethodType = "reject_out_of_network_invitations"
	ErrorMethodTypeRemoveMember                            ErrorMethodType = "remove_member"
	ErrorMethodTypeUpdateInvitePolicyByChannelID           ErrorMethodType = "update_invite_policy_by_channel_id"
	ErrorMethodTypeUpdateSquad                             ErrorMethodType = "update_squad"
)

// SquadStreamErrorTrackingEventInfo represents the information needed to fire an error tracking event.
type SquadStreamErrorTrackingEventInfo struct {
	ChannelID       string  // the channel invoking the error'd action
	TargetChannelID *string // the target of the error'd action (if there is one)
	SquadID         string
	InviteID        string
	NumMembers      *int
	Method          ErrorMethodType
	ErrorCode       string
}
