package playerstore

import (
	"context"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/twitch-events/meepo/internal/cache"
)

// Config contains the configs for the player store.
type Config struct {
	cacheDuration *distconf.Duration
}

// Load loads the cache duration.
func (c *Config) Load(d *distconf.Distconf) error {
	c.cacheDuration = d.Duration("meepo.playerstore.cache_duration", 10*time.Minute)
	return nil
}

// PlayerStore contains the cache service for the player store.
type PlayerStore struct {
	config *Config
	cache  *cache.Service

	log *log.ElevatedLog
}

// NewPlayerStore initializes the player store using the given cache service.
func NewPlayerStore(config *Config, cache *cache.Service, logger *log.ElevatedLog) (*PlayerStore, error) {
	return &PlayerStore{
		config: config,
		cache:  cache,
		log:    logger,
	}, nil
}

func cachedPrimaryPlayerChannelIDKey(deviceID string, squadID string) string {
	return "mpo:pp1:" + deviceID + ":" + squadID
}

// GetPrimaryPlayerChannelID gets the primary player channel ID per deviceID & squadID pair from cache.
func (b *PlayerStore) GetPrimaryPlayerChannelID(ctx context.Context, deviceID string, squadID string) (*string, error) {
	key := cachedPrimaryPlayerChannelIDKey(deviceID, squadID)
	cached, err := b.cache.Get(ctx, key)
	if err != nil {
		if err == memcache.ErrCacheMiss {
			return nil, nil
		}

		b.log.LogCtx(ctx, "err", err, "key", key, "could not get primary player channel ID from cache")
		return nil, err
	}
	primaryPlayerChannelID := string(cached.Value)
	return &primaryPlayerChannelID, nil
}

// SetPrimaryPlayerChannelID sets the primary player channel ID per deviceID & squadID pair in cache.
func (b *PlayerStore) SetPrimaryPlayerChannelID(ctx context.Context, deviceID string, squadID string, primaryPlayerChannelID string) error {
	key := cachedPrimaryPlayerChannelIDKey(deviceID, squadID)
	err := b.cache.Set(ctx, &memcache.Item{
		Key:        key,
		Value:      []byte(primaryPlayerChannelID),
		Expiration: int32(b.config.cacheDuration.Get().Seconds()),
	})
	if err != nil {
		b.log.LogCtx(ctx, "err", err, "key", key, "could not cache primary player channel ID")
		return err
	}
	return nil
}
