package stubs

import (
	"context"
	"errors"
	"sync"

	"code.justin.tv/twitch-events/meepo/clients"
)

// LivelineStub allows us to control the result of LivelineClient functions.
type LivelineStub struct {
	mutex sync.Mutex

	shouldError bool

	// The set of channels that liveline will return as being live.
	liveChannels map[string]bool
}

// LivelineStub implements the LivelineClient interface
var _ clients.LivelineClient = &LivelineStub{}

// GetLiveChannelsByChannelIDs checks channelIDs against l.liveChannels, and returns the subset of channelIDs that are live.
func (l *LivelineStub) GetLiveChannelsByChannelIDs(ctx context.Context, channelIDs []string) ([]string, error) {
	if l.shouldError {
		return nil, errors.New("liveline stub was set up to send an error")
	}

	liveChannelSlice := make([]string, 0)

	l.mutex.Lock()
	for k, v := range l.liveChannels {
		if v {
			liveChannelSlice = append(liveChannelSlice, k)
		}
	}
	l.mutex.Unlock()

	return liveChannelSlice, nil
}

// AddLiveChannels adds channels to our liveChannels map.
func (l *LivelineStub) AddLiveChannels(channelIDs []string) {
	l.mutex.Lock()
	for _, channelID := range channelIDs {
		l.liveChannels[channelID] = true
	}
	l.mutex.Unlock()
}

// RemoveLiveChannels removes channels from our liveChannels map.
func (l *LivelineStub) RemoveLiveChannels(channelIDs []string) {
	l.mutex.Lock()
	for _, channelID := range channelIDs {
		l.liveChannels[channelID] = false
	}
	l.mutex.Unlock()
}

// SetLiveChannels replaces our liveChannels map with a new set of live channels.
func (l *LivelineStub) SetLiveChannels(channelIDs []string) {
	l.mutex.Lock()
	l.liveChannels = make(map[string]bool)

	for _, channelID := range channelIDs {
		l.liveChannels[channelID] = true
	}
	l.mutex.Unlock()
}

// ClearLiveChannels removes all currently live channels from our liveChannels map.
func (l *LivelineStub) ClearLiveChannels() {
	l.mutex.Lock()
	l.liveChannels = make(map[string]bool)
	l.mutex.Unlock()
}

// SetShouldError sets whether or not using the Liveline Client should result in an error.
func (l *LivelineStub) SetShouldError(shouldError bool) {
	l.mutex.Lock()
	l.shouldError = shouldError
	l.mutex.Unlock()
}
