// +build integration

package testutil

import (
	"testing"

	"context"
	"strings"
	"sync"

	"code.justin.tv/twitch-events/meepo/clients"
	"code.justin.tv/twitch-events/meepo/internal/models"
)

// NewPubsubMockClient creates a PubsubMockClient.
func NewPubsubMockClient() *PubsubMockClient {
	return &PubsubMockClient{}
}

// PubsubMockClient is a handwritten mock of Meepo's pubsub client.
//
// We don't use a generated testify/mock mock because it was difficult to distinguish between calls made
// during the set-up of our test, and during the operation that we are actually trying to test.  To solve
// this problem, PubsubMockClient supports clearing recorded calls in a thread-safe way.
type PubsubMockClient struct {
	mutex sync.Mutex

	channelNotInSquadCalls    []string
	squadUpdatesCalls         []squadUpdateArgs
	incomingSquadInvitesCalls []incomingSquadInvitesArgs
}

var _ clients.PubsubClient = &PubsubMockClient{}

// ExpectedPubsubInvitations represent an invitation that a test expects Meepo publish to Pubsub.
type ExpectedPubsubInvitations struct {
	SenderID    string
	RecipientID string
	Status      models.InvitationStatus
}

type incomingSquadInvitesArgs struct {
	targetChannelID string
	invitations     []actualPubsubInvitation
}

type actualPubsubInvitation struct {
	senderID    string
	recipientID string
	status      models.InvitationStatus
}

type squadUpdateArgs struct {
	squadID string
	status  models.SquadStatus
	members []models.PubsubUser
}

// PublishChannelNotInSquad records the args in the call being made.
func (m *PubsubMockClient) PublishChannelNotInSquad(ctx context.Context, channelID string) error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	m.channelNotInSquadCalls = append(m.channelNotInSquadCalls, channelID)
	return nil
}

// PublishSquadUpdate records the args in the call being made.
func (m *PubsubMockClient) PublishSquadUpdate(ctx context.Context, managedSquad *models.PubsubManagedSquad) error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	if managedSquad == nil {
		return nil
	}

	m.squadUpdatesCalls = append(m.squadUpdatesCalls, squadUpdateArgs{
		squadID: managedSquad.ID,
		status:  managedSquad.Status,
		members: managedSquad.Members,
	})

	return nil
}

// PublishIncomingSquadInvites records the args in the call being made.
func (m *PubsubMockClient) PublishIncomingSquadInvites(ctx context.Context, recipientID string, invitations []*models.PubsubIncomingInvitation) error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	actualInvites := make([]actualPubsubInvitation, len(invitations))
	for i, inv := range invitations {
		actualInvites[i] = actualPubsubInvitation{
			senderID:    inv.Sender.ID,
			recipientID: inv.Recipient.ID,
			status:      inv.Status,
		}
	}

	m.incomingSquadInvitesCalls = append(m.incomingSquadInvitesCalls, incomingSquadInvitesArgs{
		targetChannelID: recipientID,
		invitations:     actualInvites,
	})
	return nil
}

// ClearCalls clears the recorded calls.
func (m *PubsubMockClient) ClearCalls() {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	m.channelNotInSquadCalls = nil
	m.squadUpdatesCalls = nil
	m.incomingSquadInvitesCalls = nil
}

// AssertSquadUpdatePublished asserts whether we published the updated squad to the public pubsub topic that is
// indexed by squad ID.
func (m *PubsubMockClient) AssertSquadUpdatePublished(t *testing.T, squadID string, status models.SquadStatus, memberIDs []string) {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	for _, call := range m.squadUpdatesCalls {
		if call.squadID == squadID && call.status == status && areMembersEquivalent(call.members, memberIDs) {
			return
		}
	}

	t.Fatalf("could not find call for PublishSquadUpdate where squadID: %s, status: %v, memberIDs: %s",
		squadID, status, strings.Join(memberIDs, ", "))
}

// AssertSquadUpdatePublishedTimes asserts whether we published the updated squad to the public pubsub topic n number of times
func (m *PubsubMockClient) AssertSquadUpdatePublishedTimes(t *testing.T, squadID string, status models.SquadStatus, memberIDs []string, times int) {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	count := 0
	for _, call := range m.squadUpdatesCalls {
		if call.squadID == squadID && call.status == status && areMembersEquivalent(call.members, memberIDs) {
			count++
		}
	}

	if count == times {
		return
	}

	t.Fatalf("expected %v calls for PublishSquadUpdate but got %v calls; where squadID: %s, status: %v, memberIDs: %s",
		times, count, squadID, status, strings.Join(memberIDs, ", "))
}

// AssertSquadUpdateNotPublished asserts that we did not publish this updated squad to pubsub.
func (m *PubsubMockClient) AssertSquadUpdateNotPublished(t *testing.T, squadID string, status models.SquadStatus, memberIDs []string) {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	for _, call := range m.squadUpdatesCalls {
		if call.squadID == squadID && call.status == status && areMembersEquivalent(call.members, memberIDs) {
			t.Fatalf("found call for PublishSquadUpdate where squadID: %s, status: %v, memberIDs: %s",
				squadID, status, strings.Join(memberIDs, ", "))
		}
	}
}

// AssertChannelNotInSquadPublished asserts whether we published that there is no squad.
func (m *PubsubMockClient) AssertChannelNotInSquadPublished(t *testing.T, targetChannelID string) {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	for _, channelID := range m.channelNotInSquadCalls {
		if channelID == targetChannelID {
			return
		}
	}

	t.Fatalf("could not find call for PublishChannelNotInSquad where targetChannelID: %s", targetChannelID)
}

// AssertIncomingSquadInvitesPublished asserts whether we published a channel's incoming invites to pubsub.
func (m *PubsubMockClient) AssertIncomingSquadInvitesPublished(t *testing.T, targetChannelID string, expectedInvites []ExpectedPubsubInvitations) {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	for _, call := range m.incomingSquadInvitesCalls {
		ok := m.hasIncomingSquadInvitesMatch(call, targetChannelID, expectedInvites)
		if ok {
			return
		}
	}
	t.Fatalf("could not find call for PublishIncomingSquadInvites where targetChannelID: %s", targetChannelID)
}

// AssertIncomingSquadInvitesNotPublished asserts that expectedInvites were NOT published.
func (m *PubsubMockClient) AssertIncomingSquadInvitesNotPublished(t *testing.T, targetChannelID string, expectedInvites []ExpectedPubsubInvitations) {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	for _, call := range m.incomingSquadInvitesCalls {
		ok := m.hasIncomingSquadInvitesMatch(call, targetChannelID, expectedInvites)
		if ok {
			t.Fatalf("could not find call for PublishIncomingSquadInvites where targetChannelID: %s", targetChannelID)
		}
	}
}

func (m *PubsubMockClient) hasIncomingSquadInvitesMatch(call incomingSquadInvitesArgs, targetChannelID string, expectedInvites []ExpectedPubsubInvitations) bool {
	if targetChannelID != call.targetChannelID {
		return false
	}

	if len(expectedInvites) != len(call.invitations) {
		return false
	}

	for _, expectedInvite := range expectedInvites {
		matchFound := false
		for _, actualInvite := range call.invitations {
			if actualInvite.senderID == expectedInvite.SenderID &&
				actualInvite.recipientID == expectedInvite.RecipientID &&
				actualInvite.status == expectedInvite.Status {
				matchFound = true
				break
			}
		}
		if !matchFound {
			return false
		}
	}

	return true
}

func areMembersEquivalent(members []models.PubsubUser, expectedMemberIDs []string) bool {
	if len(members) != len(expectedMemberIDs) {
		return false
	}

	memberIDSet := make(map[string]bool, len(members))
	for _, member := range members {
		memberIDSet[member.ID] = true
	}

	for _, expectedMemberID := range expectedMemberIDs {
		if !memberIDSet[expectedMemberID] {
			return false
		}
	}

	return true
}
