resource "aws_iam_role" "app" {
  name = "${var.team_name}-${var.service_name}-${var.environment}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Principal": {
        ${var.allow_assume_role_arn != "" ? "\"AWS\": \"${var.allow_assume_role_arn}\"," : ""}
        "Service": "ec2.amazonaws.com"
      }
    }
  ]
}
EOF
}

resource "aws_iam_instance_profile" "app" {
  name = "${var.team_name}-${var.service_name}-${var.environment}"
  role = aws_iam_role.app.name
}

resource "aws_iam_role_policy" "policy" {
  name       = "${var.team_name}-${var.service_name}_${var.environment}_policy"
  role       = aws_iam_role.app.id
  depends_on = [aws_iam_role.app]

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": "sts:AssumeRole",
      "Resource": "${var.sandstorm_role_arn}"
    },
    {
      "Effect": "Allow",
      "Action": "sns:Publish",
      "Resource": [
        "${aws_sns_topic.squad_channels_sns_topic.arn}"
      ]
    },
    {
      "Sid": "",
      "Effect": "Allow",
      "Action": "sts:AssumeRole",
      "Resource": [
        "arn:aws:iam::895799599216:role/PDMSLambda-CallerRole-18451FI19HSXT",
        "arn:aws:iam::125704643346:role/PDMSLambda-CallerRole-13IIND444YKVR"
      ]
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "cloudwatch_full_access_attachment" {
  role       = aws_iam_role.app.name
  policy_arn = "arn:aws:iam::aws:policy/CloudWatchLogsFullAccess"
}

# IAM user that Jenkins uses for builds and deploys.
resource "aws_iam_user" "jenkins_user" {
  name = "${var.team_name}-${var.service_name}-deploy-${var.environment}"
}

# Attach the AWSElasticBeanstalkFullAccess policy to the Jenkins user so that
# it can deploy to Elastic Beanstalk.
resource "aws_iam_user_policy_attachment" "jenkins-user-beanstalk-attachment" {
  user       = aws_iam_user.jenkins_user.name
  policy_arn = "arn:aws:iam::aws:policy/AWSElasticBeanstalkFullAccess"
}

resource "aws_iam_user_policy" "jenkins_user_sandstorm_policy" {
  name       = "${var.team_name}-${var.service_name}-deploy-${var.environment}_policy"
  user       = aws_iam_user.jenkins_user.name
  depends_on = [aws_iam_user.jenkins_user]

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": "sts:AssumeRole",
      "Resource": "${var.sandstorm_role_arn}"
    }
  ]
}
EOF
}

resource "aws_sns_topic_policy" "squad_channels_sns_topic_policy" {
  arn = aws_sns_topic.squad_channels_sns_topic.arn

  # Currently, principals allowed to subscribe are:
  # - discovery/graffiti
  policy = <<EOF
{
  "Version": "2008-10-17",
  "Id": "channel_live_policy_ID",
  "Statement": [
    {
      "Sid": "AllowSubscriptionsFromOtherTwitchAWSAccounts",
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::924637445623:root"
      },
      "Action": [
        "SNS:Subscribe",
        "SNS:Receive",
        "SNS:ListSubscriptionsByTopic"
      ],
      "Resource": "${aws_sns_topic.squad_channels_sns_topic.arn}"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "beanstalk_role_ssm_attachment" {
  role       = aws_iam_role.app.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMReadOnlyAccess"
}

resource "aws_iam_user_policy_attachment" "jenkins_user_ssm_attachment" {
  user       = aws_iam_user.jenkins_user.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMReadOnlyAccess"
}

resource "aws_iam_role_policy_attachment" "instance_role_ssm_agent_attachment" {
  role = aws_iam_role.app.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
}
