resource "aws_elastic_beanstalk_application" "app" {
  name        = "${var.name}-${var.environment}"
  description = "${var.team_display_name} team's ${var.project} service for ${var.environment}"
}

resource "aws_elastic_beanstalk_environment" "web" {
  lifecycle {
    ignore_changes = [solution_stack_name]
  }

  name        = "${var.name}-${var.environment}-web"
  application = aws_elastic_beanstalk_application.app.name

  solution_stack_name = "64bit Amazon Linux 2018.03 v2.12.9 running Docker 18.06.1-ce"
  cname_prefix        = "${var.name}-${var.environment}"

  tags = {
    Owner   = var.owner
    Project = var.project
    Team    = var.team_display_name
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "DeploymentPolicy"
    value     = "Rolling"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "BatchSizeType"
    value     = "Percentage"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "BatchSize"
    value     = "33"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application"
    name      = "Application Healthcheck URL"
    value     = "/debug/health"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name      = "SystemType"
    value     = "enhanced"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name      = "ConfigDocument"

    value = <<EOF
{
  "CloudWatchMetrics": {
    "Environment": {
      "ApplicationLatencyP50": 60,
      "ApplicationLatencyP90": 60,
      "ApplicationLatencyP99": 60,
      "ApplicationRequests2xx": 60,
      "ApplicationRequests3xx": 60,
      "ApplicationRequests4xx": 60,
      "ApplicationRequests5xx": 60,
      "ApplicationRequestsTotal": 60,
      "InstancesUnknown": 60,
      "InstancesNoData": 60,
      "InstancesSevere": 60,
      "InstancesWarning": 60,
      "InstancesDegraded": 60,
      "InstancesPending": 60,
      "InstancesInfo": 60,
      "InstancesOk": 60
    },
    "Instance": {
      "ApplicationLatencyP50": 60,
      "ApplicationLatencyP90": 60,
      "ApplicationLatencyP99": 60,
      "ApplicationRequests2xx": 60,
      "ApplicationRequests3xx": 60,
      "ApplicationRequests4xx": 60,
      "ApplicationRequests5xx": 60,
      "ApplicationRequestsTotal": 60,
      "CPUIdle": 60,
      "CPUIowait": 60,
      "CPUIrq": 60,
      "CPUNice": 60,
      "CPUSoftirq": 60,
      "CPUSystem": 60,
      "CPUUser": 60,
      "InstanceHealth": 60,
      "LoadAverage1min": 60,
      "LoadAverage5min": 60,
      "RootFilesystemUtil": 60
    }
  },
  "Rules": {
    "Environment": {
      "Application": {
        "ApplicationRequests4xx": {
          "Enabled": false
        }
      }
    }
  },
  "Version": 1
}
EOF
  }

  setting {
    namespace = "aws:elasticbeanstalk:environment"
    name      = "ServiceRole"
    value     = "aws-elasticbeanstalk-service-role"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "ENVIRONMENT"
    value     = var.environment
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "SANDSTORM_IAM_ROLE_ARN"
    value     = var.sandstorm_role_arn
  }

  // Elastic Load Balance
  setting {
    namespace = "aws:elb:loadbalancer"
    name      = "CrossZone"
    value     = "true"
  }

  setting {
    namespace = "aws:elb:healthcheck"
    name      = "Interval"
    value     = "30"
  }

  setting {
    namespace = "aws:elb:policies"
    name      = "ConnectionDrainingEnabled"
    value     = "true"
  }

  // Autoscaling Group
  setting {
    namespace = "aws:autoscaling:asg"
    name      = "Availability Zones"
    value     = "Any 2"
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name      = "MinSize"
    value     = var.asg_min_size
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name      = "MaxSize"
    value     = var.asg_max_size
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "MeasureName"
    value     = "CPUUtilization"
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "LowerThreshold"
    value     = "40"
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "UpperThreshold"
    value     = "60"
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "Unit"
    value     = "Percent"
  }

  // Launch configuration
  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "IamInstanceProfile"
    value     = aws_iam_instance_profile.app.id
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "SecurityGroups"
    value     = join(",", var.twitch_subnets_sg)
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "EC2KeyName"
    value     = var.aws_key_name
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "InstanceType"
    value     = var.instance_type
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name      = "RollingUpdateEnabled"
    value     = "true"
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name      = "RollingUpdateType"
    value     = "Health"
  }

  // EC2 VPC
  setting {
    namespace = "aws:ec2:vpc"
    name      = "VPCId"
    value     = var.vpc_id
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "Subnets"
    value     = join(",", var.private_subnets)
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "AssociatePublicIpAddress"
    value     = "false"
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "ELBScheme"
    value     = "internal"
  }

  // Cloudwatch
  setting {
    namespace = "aws:elasticbeanstalk:cloudwatch:logs"
    name      = "StreamLogs"
    value     = "true"
  }

  // Managed updates settings
  setting {
    namespace = "aws:elasticbeanstalk:managedactions"
    name      = "ManagedActionsEnabled"
    value     = "true"
  }

  setting {
    namespace = "aws:elasticbeanstalk:managedactions"
    name      = "PreferredStartTime"
    value     = "Tue:09:00"
  }

  setting {
    namespace = "aws:elasticbeanstalk:managedactions:platformupdate"
    name      = "UpdateLevel"
    value     = "minor"
  }
}
