package rollbar

import (
	"os"
	"runtime"
	"strings"
)

func buildStack(skip int, err error) []Frame {
	if f := stackTraceFromError(err); f != nil {
		return f
	}
	s := make([]Frame, 0, 64)

	for i := skip; ; i++ {
		pc, file, line, ok := runtime.Caller(i)
		if !ok {
			break
		}
		file = shortenFilePath(file)
		s = append(s, Frame{Filename: file, Method: functionName(pc), Line: line})
	}

	return s
}

func functionName(pc uintptr) string {
	fn := runtime.FuncForPC(pc)
	if fn == nil {
		return "???"
	}
	name := fn.Name()
	end := strings.LastIndex(name, string(os.PathSeparator))
	return name[end+1:]
}

func shortenFilePath(s string) string {
	idx := strings.Index(s, "/src/")
	if idx != -1 {
		return s[idx+5:]
	}
	return s
}

func fromPointers(frames []uintptr) []Frame {
	retFrames := make([]Frame, 0, len(frames))
	for _, p := range frames {
		// I honestly don't know why we have to -1 .  But it's what https://github.com/pkg/errors/blob/master/stack.go does?
		p = p - 1
		f := runtime.FuncForPC(p)
		if f == nil {
			retFrames = append(retFrames, Frame{
				Filename: "???",
				Method:   "???",
				Line:     0,
			})
			continue
		}
		file, l := f.FileLine(p)
		retFrames = append(retFrames, Frame{
			Filename: file,
			Method:   f.Name(),
			Line:     l,
		})
	}
	return retFrames
}

func stackTraceFromError(err error) []Frame {
	type tracedError1 interface {
		StackTrace() []uintptr
	}
	type tracedError2 interface {
		Stack() []uintptr
	}
	if s1, ok := err.(tracedError1); ok {
		return fromPointers(s1.StackTrace())
	}
	if s1, ok := err.(tracedError2); ok {
		return fromPointers(s1.Stack())
	}
	return nil
}
