package rolllog

import (
	"fmt"
	"reflect"
	"sync"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/rollbar"
	"golang.org/x/net/context"
)

type Config struct {
	AccessToken *distconf.Str
	SendTimeout *distconf.Duration
}

func (c *Config) Load(dconf *distconf.Distconf) {
	c.AccessToken = dconf.Str("rollbar.access_token", "")
	c.SendTimeout = dconf.Duration("rollbar.send_timeout", time.Second*3)
}

func (c *Config) Monitor(r *Rolllog) {
	r.Update(c)
	c.AccessToken.Watch(func() {
		r.Update(c)
	})
	c.SendTimeout.Watch(func() {
		r.Update(c)
	})
}

type Rolllog struct {
	OnErr           func(error)
	Client          *rollbar.Client
	DefaultErrLevel rollbar.Level
	DefaultMsgLevel rollbar.Level
	SendTimeout     time.Duration
	mu              sync.RWMutex
}

func (r *Rolllog) Update(conf *Config) {
	r.mu.Lock()
	defer r.mu.Unlock()

	r.Client.AccessToken = conf.AccessToken.Get()
	r.SendTimeout = conf.SendTimeout.Get()
}

func (r *Rolllog) Log(vals ...interface{}) {
	r.mu.RLock()
	defer r.mu.RUnlock()
	if r.Client.AccessToken == "" {
		return
	}
	ctx := context.Background()
	if r.SendTimeout != 0 {
		var onCancel context.CancelFunc
		ctx, onCancel = context.WithTimeout(ctx, r.SendTimeout)
		defer onCancel()
	}
	optionals := make(map[string]string, len(vals)/2+1)
	msg := "<empty>"
	var err error
	level := rollbar.Level("")

	for i := 0; i < len(vals); i += 2 {
		if i == len(vals)-1 {
			msg = mapKey(vals[i])
			continue
		}
		prevKey := mapKey(vals[i])
		if prevKey == "err" {
			if nextAsErr, ok := vals[i+1].(error); ok {
				err = nextAsErr
				continue
			}
		}
		if prevKey == "level" {
			level = rollbar.Level(mapKey(vals[i+1]))
			continue
		}
		if prevKey == "msg" {
			msg = mapKey(vals[i+1])
			continue
		}
		optionals[mapKey(vals[i])] = mapKey(vals[i+1])
	}
	if err != nil {
		r.sendError(ctx, level, err, optionals)
		return
	}
	r.sendMessage(ctx, level, msg, optionals)
}

func (r *Rolllog) sendError(ctx context.Context, level rollbar.Level, err error, optionals map[string]string) {
	if level == "" {
		level = r.DefaultErrLevel
	}
	d := rollbar.TraceBody(level, err)
	d.Fingerprint = rollbar.StackFrameFingerprint(d.Body.Trace.Frames)
	d.Custom = optionals
	if _, sendErr := r.Client.Send(ctx, d); sendErr != nil {
		r.OnErr(sendErr)
	}
}

func (r *Rolllog) sendMessage(ctx context.Context, level rollbar.Level, msg string, optionals map[string]string) {
	if level == "" {
		level = r.DefaultMsgLevel
	}
	d := rollbar.MessageBody(level, msg)
	d.Custom = optionals
	if _, sendErr := r.Client.Send(ctx, d); sendErr != nil {
		r.OnErr(sendErr)
	}
}

// Different from go-kit.  People just shouldn't pass nil values and should know if they do
func mapKey(k interface{}) (s string) {
	defer func() {
		if panicVal := recover(); panicVal != nil {
			s = nilCheck(k, panicVal, "NULL").(string)
		}
	}()
	switch x := k.(type) {
	case string:
		return x
	case fmt.Stringer:
		return x.String()
	default:
		return fmt.Sprint(x)
	}
}

func nilCheck(ptr, panicVal interface{}, onError interface{}) interface{} {
	if vl := reflect.ValueOf(ptr); vl.Kind() == reflect.Ptr && vl.IsNil() {
		return onError
	}
	panic(panicVal)
}
