<?php
namespace App\Http\Controllers;

use Carbon\CarbonPeriod;
use GuzzleHttp\Client;
use Illuminate\Support\Carbon;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Subscriber\Oauth\Oauth1;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;

define('EXPIRY', 5); //how many minutes?)
define('DEBUG', TRUE); //how many minutes?)
date_default_timezone_set('America/Los_Angeles');
ini_set("auto_detect_line_endings", "1");

//Look back at launched items for the current year
class FVBurndown extends BaseController
{
    public $fixVersion = null;
    public $metricType;
    public $statuses = array();

    function __construct()
    {
        $this->credentials = include('../../jira-credentials.php');
        $this->credentialsString = $this->credentials['user'] . ":" . $this->credentials['password'];
    }

    //pull data from JIRA greenhopper rest calls
    //https://jira.xarth.tv/rest/greenhopper/1.0/rapid/charts/versionprogresschart?rapidViewId=904&versionId=32433&_=1558737554986
    //https://jira.xarth.tv/rest/greenhopper/1.0/rapid/charts/releaseburndownchart?rapidViewId=904&versionId=32433&_=1558739260968
    //https://jira.xarth.tv/rest/greenhopper/1.0/rapid/charts/releaseburndownchart?rapidViewId=904&versionId=33411&_=1559936508248
    public function getJiraBurndownData($project = "ce", $subproject = "sunlight", $jiraproject = "tcp") {
        $stack = HandlerStack::create();

        $middleware = new Oauth1([
            'consumer_key'    => 'twitchtpm',
            'consumer_secret' => '',
            'token' => env("JIRA_OAUTH_TOKEN"),
            'token_secret' => env("JIRA_OAUTH_SECRET"),
            'private_key_file' => '../../privkey.pem',
            'private_key_passphrase' => '',
            'signature_method' => Oauth1::SIGNATURE_METHOD_RSA,
        ]);
        $stack->push($middleware);

        $client = new Client([
            'base_uri' => 'https://jira.xarth.tv/rest/greenhopper/1.0/rapid/charts/',
            'handler' => $stack,
            'auth' => 'oauth'
        ]);

        $today = date("Ymd");
        $modelReflector = new \ReflectionClass(__CLASS__);
        $method = $modelReflector->getMethod(__FUNCTION__)->getName();
        $key = md5($method . $project . $subproject . $jiraproject . $today);

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(EXPIRY);
            $response = $client->get(Config::get("$project.$subproject.burndowns.$jiraproject.burndownUri"));
            $results = json_decode($response->getBody()->getContents());
            Cache::add($key, $results, $expiresAt);
        }

//        dd($results);
        //release burndown
        if ($results != null && isset($results->version)) {
            $this->fixVersion = $results->version; //save fix version info
//            dd($this->fixVersion);
        } else {
        //epic burndown pull from config
            $this->fixVersion = new \stdClass();
            $this->fixVersion->startDate = $results->epicCreationTime;
            $this->fixVersion->startDate = Config::get("$project.$subproject.burndowns.$jiraproject.startDate");
            $this->fixVersion->releaseDate = Config::get("$project.$subproject.burndowns.$jiraproject.releaseDate");
        }
        if ($results != null && $results->statisticField) {
            $this->metricType = $results->statisticField->renderer;
        }
        return $results;
    }

    //https://jira.xarth.tv/rest/greenhopper/1.0/rapid/charts/versionreport?rapidViewId=904&versionId=33411&_=1560378800405
    public function getJiraStatuses($project = "ce", $subproject = "sunlight", $jiraproject = "tcp") {
        $stack = HandlerStack::create();

        $middleware = new Oauth1([
            'consumer_key'    => 'twitchtpm',
            'consumer_secret' => '',
            'token' => env("JIRA_OAUTH_TOKEN"),
            'token_secret' => env("JIRA_OAUTH_SECRET"),
            'private_key_file' => '../../privkey.pem',
            'private_key_passphrase' => '',
            'signature_method' => Oauth1::SIGNATURE_METHOD_RSA,
        ]);
        $stack->push($middleware);

        $client = new Client([
            'base_uri' => 'https://jira.xarth.tv/rest/greenhopper/1.0/rapid/charts/',
            'handler' => $stack,
            'auth' => 'oauth'
        ]);

        $today = date("Ymd");
        $modelReflector = new \ReflectionClass(__CLASS__);
        $method = $modelReflector->getMethod(__FUNCTION__)->getName();
        $key = md5($method . $project . $subproject . $jiraproject . $today);

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(EXPIRY);
            $response = $client->get(Config::get("$project.$subproject.burndowns.$jiraproject.statusUri"));
            $results = json_decode($response->getBody()->getContents());
            Cache::add($key, $results, $expiresAt);
        }

        foreach($results->doneStatuses as $status) {
            $this->statuses["Closed"][] = $status->id;
        }

        foreach($results->notDoneStatuses as $status) {
            $this->statuses["Work Remaining"][] = $status->id;
        }
//        dd($this->statuses);
        return $this->statuses;
    }


//    doneStatuses:
//    0: {name: "Done", id: "10007"}
//    1: {name: "Closed", id: "6"}
//    notDoneStatuses:
//    0: {name: "Approved", id: "11908"}
//    1: {name: "Product Verify", id: "17300"}
//    2: {name: "Review", id: "12407"}
//    3: {name: "Rejected", id: "11910"}
//    4: {name: "In Progress", id: "3"}
//    5: {name: "Awaiting Manager Approval", id: "16500"}
//    6: {name: "Backlog", id: "11107"}
//    7: {name: "On Deck / Ready", id: "11108"}
//    8: {name: "Open", id: "1"}
    public function getBurndownLedgerData($project = "ce", $subproject = "sunlight", $jiraproject = "tcp") {
        $this->getJiraStatuses($project, $subproject, $jiraproject); //initialize status look ups
        $results = $this->getJiraBurndownData($project, $subproject, $jiraproject);
//        dd($results->changes);
        $changes = array();
        $storyPoints = array();
        foreach($results->changes as $time => $change) {
//            dd($change);
            $date = Carbon::createFromTimestampMs($time, "UTC")->setTimezone("America/Los_Angeles")->format("Ymd");

            //has initial status to determine if it's Work Remaining or Closed
            if (isset($change[0]->column)) {
//                dd($change);
                //bucket all the statuses into 1 of the two b/c there are a lot of different ones
                if (isset($change[0]->column->newStatus) && in_array($change[0]->column->newStatus, $this->statuses["Closed"])) {
                    $status = "Closed";
                } else {
                    $status = "Work Remaining";
                }


                if (isset($change[0]->statC->oldValue)) {
                    //scope reduced
                    $changes[$date][$status][$change[0]->key] = $change[0]->statC->oldValue;
                    $storyPoints[$change[0]->key] = $change[0]->statC->oldValue;
                } else if (isset($change[0]->statC->newValue)) {
                    //initialize and has scope
                    $changes[$date][$status][$change[0]->key] = $change[0]->statC->newValue;
                    $storyPoints[$change[0]->key] = $change[0]->statC->newValue;
                } else {
                    //initialize but has no scope
                    if (isset($storyPoints[$change[0]->key])) {
                        $changes[$date][$status][$change[0]->key] = $storyPoints[$change[0]->key];
                    } else {
                        $changes[$date][$status][$change[0]->key] = 0;
                    }
                }
            } else if (isset($change[0]->statC->newValue)) {
                //scope added after the fact so assume Work Remaining
                $changes[$date]["Work Remaining"][$change[0]->key] = $change[0]->statC->newValue;
                $storyPoints[$change[0]->key] = $change[0]->statC->newValue;
            } else if (isset($change[0]->added) && $change[0]->added == false) {
                //work entirely removed from fix version
                $changes[$date]["Closed"][$change[0]->key] = 0;
                $storyPoints[$change[0]->key] = 0;
            }
        }
//        dd($changes);
//        dd($storyPoints);
        return $changes;
    }

    public function convertToGraphData($project = "ce", $subproject = "sunlight", $jiraproject = "tcp") {
        $ledger = $this->getBurndownLedgerData($project, $subproject, $jiraproject); //raw ledger
//        dd($ledger);
        $timeline = array();
        $prevDate = array_key_first($ledger);
        foreach ($ledger as $date => $statuses) {
            foreach ($statuses as $status => $stories) {
                $timeline[$date][$status] = $stories;
            }

            if ($prevDate != $date) {
                foreach ($timeline[$prevDate] as $status => $values) {
                    if (isset($timeline[$date][$status])) {
                        $timeline[$date][$status] = array_merge($timeline[$prevDate][$status], $timeline[$date][$status]);
                    } else if (isset($timeline[$date]["Closed"])) {
                        //                        dd($timeline[$date]["Closed"]);
                        //                        dd($timeline[$prevDate]["Closed"]);
                        //                        dd($timeline[$prevDate]);
                        //                        dd(array_intersect_key($timeline[$date]["Closed"], $timeline[$prevDate][$status]));
                        //                        dd(array_diff_key($timeline[$prevDate]["Work Remaining"], $timeline[$date]["Closed"]));
                        $timeline[$date]["Work Remaining"] = array_diff_key($timeline[$prevDate]["Work Remaining"], $timeline[$date]["Closed"]);
                    } else {
                        $timeline[$date][$status] = $timeline[$prevDate][$status];
                    }
                }
            }
            $prevDate = $date;
        }

        $period = CarbonPeriod::create(Carbon::createFromFormat("Ymd", array_key_first($timeline)), Carbon::createFromFormat("Ymd", array_key_last($timeline)));
        for($i = $period->getStartDate(); $i < $period->getEndDate(); $i->addDays(1) ) {
            if (!isset($timeline[$i->format("Ymd")])) {
                $timeline[$i->format("Ymd")] = $timeline[$i->subDays(1)->format("Ymd")];

            }
        }
        ksort($timeline);


        if(!isset($this->fixVersion->startDate)) {
            $this->fixVersion->startDate = (int) Carbon::now()->subWeek(4)->valueOf(); //Millisecond
        }

        if(!isset($this->fixVersion->releaseDate)) {
            $this->fixVersion->releaseDate = (int) Carbon::now()->addWeek(5)->valueOf(); //Millisecond
        }

        foreach($timeline as $date => $status) {
            if(Carbon::createFromFormat("Ymd", $date)->greaterThan(Carbon::createFromTimestampMs($this->fixVersion->startDate))) {
                if (isset($status["Closed"])) {
                    //make sure no duplicate stories that exists in two states
                    $timeline[$date]["Work Remaining"] = array_diff_key($status["Work Remaining"], $status["Closed"]);
                }
            } else{
                //remove dates before the fix version
                unset($timeline[$date]);
            }
        }
//        dd($timeline);
        if(count($timeline) > 0) {
            //pad data until the end of the fix version
            $lastTrackedDay = Carbon::createFromFormat("Ymd", array_key_last($timeline));
            for ($i = $lastTrackedDay->addDay(1); $i->lessThan(Carbon::createFromTimestampMs($this->fixVersion->releaseDate)); $i->addDay(1)) {
                $timeline[$i->format("Ymd")]["Work Remaining"] = array();
                $timeline[$i->format("Ymd")]["Closed"] = array();

            }
            $this->createGraph($project = "ce", $subproject = "sunlight", $jiraproject = "tcp", $timeline);
        } else {
            return array();
        }
    }

    //function create target for google charts burndown
    public function createTargetData($project = "ce", $subproject = "sunlight", $jiraproject = "tcp", $array) {
        $startDate =  Carbon::createFromFormat("Ymd", array_key_first($array));
        $endDate =  Carbon::createFromFormat("Ymd", array_key_last($array));
        if($this->metricType == "duration") {
            $startingWork = array_sum($array[array_key_first($array)]["Work Remaining"]) / 3600 / 8;
        } else {
            $startingWork = array_sum($array[array_key_first($array)]["Work Remaining"]);
        }
//        dd($startDate);
//        dd($endDate);
//        dd($startingWork);

        $holidays = Config::get("$project.$subproject.projectinfo.holidays");
        Carbon::macro('isHoliday', function () use ($holidays) {
            // compatibility chunk
            if (!isset($self) && isset($this)) {
                $self = $this;
            }
            return in_array($self->format('m-d-Y'), $holidays);
        });

//        var_dump(Carbon::create(2019, 7, 1)->isHoliday());  // bool(false)
//        var_dump(Carbon::create(2019, 7, 4)->isHoliday());  // bool(true)
//        var_dump(Carbon::create(2019, 7, 5)->isHoliday());  // bool(true)

        //how many working days are there
        $numWorkingDays = $startDate->diffInDaysFiltered(function (Carbon $date) use ($holidays) {
            return $date->isWeekday() && !$date->isHoliday();
        }, $endDate) + 1;
//        dd($numWorkingDays);

        //slope required to hit 0 by end date
        $slope = round($startingWork / $numWorkingDays, 4);

        //init target array and remove first element
        $previous = array_key_first($array);
        $target[$previous] = $startingWork;
        unset($array[$previous]);

        foreach ($array as $key => $var) {
            $date = Carbon::createFromFormat("Ymd", $key);
            if ($date->isWeekday() && !$date->isHoliday()) {
                if(round($target[$previous] - $slope, 4) < 0) {
                    $target[$key] = 0;
                } else {
                    $target[$key] = round($target[$previous] - $slope, 4);
                }
            } else {
                $target[$key] = $target[$previous];
            }
            $previous = $key;
        }
//        dd($target);
        return $target;
    }

    //create google charts graph data that can be consumed via REST
    public function createGraph($project = "ce", $subproject = "sunlight", $jiraproject = "tcp", $array) {
        $table['cols'][] = array('id' => "Date", 'label' => "Date", 'type' => 'date');
        $table['cols'][] = array('id' => "Work", 'label' => "Work Remaining", 'type' => 'number');
        $table['cols'][] = array('id' => "Closed", 'label' => "Closed", 'type' => 'number');

        //target burndown
        $table['cols'][] = array('id' => "Target", 'label' => "Target", 'type' => 'number');
        $target = $this->createTargetData($project = "ce", $subproject = "sunlight", $jiraproject = "tcp", $array);

        $rows = $this->createRowDataForGoogleChart($array, $target);
        // populate the table with rows of data
        $table['rows'] = $rows;
        // encode the table as JSON
        $jsonTable = json_encode($table, JSON_NUMERIC_CHECK);
        echo $jsonTable;
    }

    //helper function to create row data for Google Chart
    //added target date + x,y names
    public function createRowDataForGoogleChart($array, $target) {
        $rows = array();

        foreach($array as $key => $values) {
            $temp = array();
            $date = Carbon::createFromFormat("Ymd", $key);
            $year = date("Y", strtotime($date));
            $month = date("m", strtotime($date)) - 1; //pass to javascript so need to subtract 1 because months start at 0 for January
            $day = date("d", strtotime($date));
            $jsdate = "Date($year, $month, $day)";
            $temp[] = array('v' => $jsdate, 'f' => date("D M d, Y", strtotime($date . "+0 day")));

            if(isset($values["Work Remaining"])) {
                if($this->metricType == "duration") {
                    //8-hour days
                    $temp[] = array('v' => array_sum($values["Work Remaining"])/3600/8);
                } else {
                    $temp[] = array('v' => array_sum($values["Work Remaining"]));
                }
            } else {
                $temp[] = array('v' => 0);
            }

            if(isset($values["Closed"])) {
                if($this->metricType == "duration") {
                    //8-hour days
                    $temp[] = array('v' => array_sum($values["Closed"])/3600/8);
                } else {
                    $temp[] = array('v' => array_sum($values["Closed"]));
                }
            } else {
                $temp[] = array('v' => 0);
            }

            //target data
            $temp[] = array('v' => $target[$key]);

            $rows[] = array('c' => $temp);
        }

        return $rows;
    }
}