<?php
namespace App\Http\Controllers;
use App\JiraTicket;
use GuzzleHttp\Client;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cache;
use Illuminate\Http\Request;

if (!defined('EXPIRY')) define('EXPIRY', 5); 
define('DEBUG', TRUE); 
date_default_timezone_set('America/Los_Angeles');

class Spotlight extends BaseController
{
    //Spotlight controller for use with spotlight.blade.php View
    public function showSpotlightItem($project = "spotlight")
    {
        $template = "spotlight";
        return View::make($template,
            array(
                'project' => $project,
                'projectName' => Config::get($project . '.projectinfo.name'), 
                'epic' => $this->getSpotlightEpic($project), // Epic from JIRA to pull in program status
                'allIssuesInEpic' => $this->getSpotlightEpicIssues($project), // All issues from JIRA to calc % complete TODO: Can probably combine this with the other queries.
                'allIssues' => $this->getSpotlightAllIssues($project), // All issues from Spotlight to calc % complete
                'emailSummary' => $this->getEmailSummary($project), // Custom narrative from Google Sheets
                'blocked' => $this->getSpotlightBlockedItems($project), // Blocked tickets from JIRA
                'planned' => $this->getSpotlightPlannedItems($project), // Planned for this sprint from JIRA
                'doneIssues' => $this->getSpotlightDoneItems($project), // Completed issues from JIRA
                'createdIssues' => $this->getSpotlightCreatedItems($project), // Created in the last 2weeks
                'contacts' => Config::get($project .'.contacts') // Footer content
                )
        );
    }

    //send test or final email to address set in .env
    public function sendSpotlightEmail($type="test"){
        $project = "spotlight";
        $template = "spotlight";
        $dateframe = Carbon::today()->startOfWeek()->subWeeks(2)->format('M d') . " - " . Carbon::today()->endOfWeek()->subWeeks(1)->format('M d');
       
        $data = array(
            'project' => $project,
            'projectName' => Config::get($project . '.projectinfo.name'), 
            'epic' => $this->getSpotlightEpic($project), // Epic from JIRA to pull in program status
            'allIssuesInEpic' => $this->getSpotlightEpicIssues($project), // All issues from JIRA to calc % complete TODO: Can probably combine this with the other queries.
            'allIssues' => $this->getSpotlightAllIssues($project), // All issues from Spotlight to calc % complete
            'emailSummary' => $this->getEmailSummary($project), // Custom narrative from Google Sheets
            'blocked' => $this->getSpotlightBlockedItems($project), // Blocked tickets from JIRA
            'planned' => $this->getSpotlightPlannedItems($project), // Planned for this sprint from JIRA
            'doneIssues' => $this->getSpotlightDoneItems($project), // Completed issues from JIRA
            'createdIssues' => $this->getSpotlightCreatedItems($project), // Created in the last 2weeks
            'contacts' => Config::get($project .'.contacts') // Footer content
     );

        //send email
        if ($type == "test") {
            //TEST EMAIL
            $toEmails = explode(',', env('MAIL_TO_TEST_SPOTLIGHT'));
            \Illuminate\Support\Facades\Mail::send('spotlight', $data, function ($message) use ($project, $dateframe, $toEmails) {
//            \Illuminate\Support\Facades\Mail::send('spotlight', $data, function ($message) use ($project, $dateframe) {
                $emailSubject = env('MAIL_SUBJECT_PREPEND_TEST') . Config::get($project . '.projectinfo.name') . " (" . $dateframe . ")"; // [Spotlight] Project Status (Apr 01 - Apr 14)
                $message->to($toEmails)->subject($emailSubject);
//                $message->to("rpark@justin.tv")->subject($emailSubject);
//                $message->cc("rpark@twitch.tv")->subject($emailSubject);
                $message->from(env('MAIL_FROM'),env('MAIL_FROM_NAME'));
            });
        } elseif ($type == "final") {
            //FINAL EMAIL
//            $toEmails = explode(',', env('MAIL_TO_SPOTLIGHT'));
//            \Illuminate\Support\Facades\Mail::send('spotlight', $data, function ($message) use ($project, $toEmails, $dateframe) {
            \Illuminate\Support\Facades\Mail::send('spotlight', $data, function ($message) use ($project, $dateframe) {
               $emailSubject = Config::get($project . '.projectinfo.name') . " (" . $dateframe . ")";
//                $message->to($toEmails)->subject($emailSubject);
                $message->to("spotlight@justin.tv")->subject($emailSubject);
                $message->cc("justprogrammanagers@twitch.tv")->subject($emailSubject);
                $message->from(env('MAIL_FROM'),env('MAIL_FROM_NAME'));
            });
        }
        return View::make($template,$data);
    }

    //Get epic info
    public function getSpotlightEpic($project = "spotlight")
    {
        $spotlightJql = Config::get($project .'.jira.epicJQL');
        $consumerKey = Config::get($project .'.jira.consumerKey');
        $oauthToken = Config::get($project . '.jira.oauthToken');
        $oauthTokenSecret = Config::get($project . '.jira.oauthTokenSecret');
        $baseURI = Config::get($project . '.jira.baseURI');

        $today = date("Ymd");
        $key = "spotlightepic" . $today . "_" . $project;

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(30);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $spotlightJql));
            Cache::add($key, $results, $expiresAt);
        }
//dd($results->issues);
        if(isset($results->issues)) {
            return $results->issues;
        } else {
            return array();
        }
    }

    //Get all Spotlight issues in specific epic and calc % complete. Should match what's in https://jira.xarth.tv/secure/Dashboard.jspa?selectPageId=31659
    public function getSpotlightEpicIssues($project = "spotlight")
    {
        $alljql = Config::get($project .'.jira.spotlightInEpicJQL');
        $consumerKey = Config::get($project .'.jira.consumerKey');
        $oauthToken = Config::get($project . '.jira.oauthToken');
        $oauthTokenSecret = Config::get($project . '.jira.oauthTokenSecret');
        $baseURI = Config::get($project . '.jira.baseURI');

        $today = date("Ymd");
        $key = "spotlightInEpic" . $today . "_" . $project;

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(30);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $alljql));
            Cache::add($key, $results, $expiresAt);
        }

        if(isset($results->issues)) {
            $unresolved = 0;
            $resolved = 0;
            foreach ($results->issues as $issue) {
                
                if ($issue->fields->resolution == null)
                    $unresolved++;
                else
                    $resolved++;
            }
            $total = $unresolved + $resolved;
            $percentcomplete = round(($resolved/$total)*100);
 //           dd($results);
            return array($percentcomplete, $resolved, $total);
        } else {
            return array();
        }
    }

    //Get all Spotlight issues and calc % complete. Should match what's in https://jira.xarth.tv/secure/Dashboard.jspa?selectPageId=31659
    public function getSpotlightAllIssues($project = "spotlight")
    {
        $alljql = Config::get($project .'.jira.spotlightallJQL');
        $consumerKey = Config::get($project .'.jira.consumerKey');
        $oauthToken = Config::get($project . '.jira.oauthToken');
        $oauthTokenSecret = Config::get($project . '.jira.oauthTokenSecret');
        $baseURI = Config::get($project . '.jira.baseURI');

        $today = date("Ymd");
        $key = "spotlightall" . $today . "_" . $project;

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(30);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $alljql));
            Cache::add($key, $results, $expiresAt);
        }

        if(isset($results->issues)) {
            $unresolved = 0;
            $resolved = 0;
            foreach ($results->issues as $issue) {
                
                if ($issue->fields->resolution == null)
                    $unresolved++;
                else
                    $resolved++;
            }
            $total = $unresolved + $resolved;
            $percentcomplete = round(($resolved/$total)*100);
 //           dd($results);
            return array($percentcomplete, $resolved, $total);
        } else {
            return array();
        }
    }

    //Get email summary from Google Sheets
    public function getEmailSummary($project) {
        if(Config::get($project . ".projectinfo.summary")) {
            $sheetID = Config::get($project . '.projectinfo.summary.sheetId');
            $sheetRange = Config::get($project . '.projectinfo.summary.sheetRange');
            $emailSummarySection = \GoogleSheets::getGSheetData($sheetID, $sheetRange); 
            if(isset($emailSummarySection[Carbon::today()->format('n/j/Y')])) // if a row with today's date exists in the sheet, return that text
            {
                return $emailSummary=$emailSummarySection[Carbon::today()->format('n/j/Y')];
            }
            else { // by default, just pull in the placeholder text
                return $emailSummary=$emailSummarySection["5/3/2019"];
            }
        }
    }

    //Get Channel Page TTV from Google Sheets
    public function getChannelPageTTV($project) {
        if(Config::get($project . ".projectinfo.ttvMetrics")) {
            $sheetID = Config::get($project . '.projectinfo.ttvMetrics.sheetId');
            $sheetRange = Config::get($project . '.projectinfo.ttvMetrics.sheetRange');
//            return $ttvMetrics = \GoogleSheets::getGSheetData($sheetID, $sheetRange, 6);
            $ttvMetrics = \GoogleSheets::getGSheetData($sheetID, $sheetRange, 6);
            if(isset($ttvMetrics)) // if a sheet with Monday's date exists in the sheet, return that text
            {
                return $ttvMetrics = \GoogleSheets::getGSheetData($sheetID, $sheetRange, 6);
            }
            else {
                return $ttvMetrics=null;
            }
         }
    }

    //Get Spotlight's blocked tickets. Should match what's in https://jira.xarth.tv/secure/Dashboard.jspa?selectPageId=31804#
    public function getSpotlightBlockedItems($project = "spotlight")
    {
        $blockedjql = Config::get($project .'.jira.spotlightblockedJQL');
        $consumerKey = Config::get($project .'.jira.consumerKey');
        $oauthToken = Config::get($project . '.jira.oauthToken');
        $oauthTokenSecret = Config::get($project . '.jira.oauthTokenSecret');
        $baseURI = Config::get($project . '.jira.baseURI');

        $today = date("Ymd");
        $key = "spotlightblocked" . $today . "_" . $project;

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(30);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $blockedjql));
            Cache::add($key, $results, $expiresAt);
        }
//        dd($results);
        if(isset($results->issues)) {
            return $results->issues;
        } else {
            return array();
        }
    }

    //Get Spotlight's tickets in next/active sprint. Should match what's in https://jira.xarth.tv/secure/Dashboard.jspa?selectPageId=31804#
    public function getSpotlightPlannedItems($project = "spotlight")
    {
        $plannedjql = Config::get($project .'.jira.spotlightplannedJQL');
        $consumerKey = Config::get($project .'.jira.consumerKey');
        $oauthToken = Config::get($project . '.jira.oauthToken');
        $oauthTokenSecret = Config::get($project . '.jira.oauthTokenSecret');
        $baseURI = Config::get($project . '.jira.baseURI');

        $today = date("Ymd");
        $key = "spotlightplanned" . $today . "_" . $project;

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(30);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $plannedjql));
            Cache::add($key, $results, $expiresAt);
        }
//dd($results);
        if(isset($results->issues)) {
            return $results->issues;
        } else {
            return array();
        }
    }

    //Get Spotlight's completed tickets done in the last 2w
    public function getSpotlightDoneItems($project = "spotlight")
    {
        $donejql = Config::get($project .'.jira.spotlightdoneJQL');
        $consumerKey = Config::get($project .'.jira.consumerKey');
        $oauthToken = Config::get($project . '.jira.oauthToken');
        $oauthTokenSecret = Config::get($project . '.jira.oauthTokenSecret');
        $baseURI = Config::get($project . '.jira.baseURI');

        $today = date("Ymd");
        $key = "spotlightdone" . $today . "_" . $project;

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(30);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $donejql));
            Cache::add($key, $results, $expiresAt);
        }
        
        if(isset($results->issues)) {
            $storycount=0;
            $bugcount=0;
            foreach($results->issues as $issue) {
                if("Bug" == $issue->fields->issuetype->name) {
                    $bugcount++;
                }
                else {
                    $storycount++;
                }
            }
            return array(count($results->issues), $storycount, $bugcount);
        }
    }

    //Get Spotlight's created tickets in the last 2w
    public function getSpotlightCreatedItems($project = "spotlight")
    {
        $donejql = Config::get($project .'.jira.spotlightlast2wJQL');
        $consumerKey = Config::get($project .'.jira.consumerKey');
        $oauthToken = Config::get($project . '.jira.oauthToken');
        $oauthTokenSecret = Config::get($project . '.jira.oauthTokenSecret');
        $baseURI = Config::get($project . '.jira.baseURI');

        $today = date("Ymd");
        $key = "spotlightcreated" . $today . "_" . $project;

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(30);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $donejql));
            Cache::add($key, $results, $expiresAt);
        }
        //dd($results);
        if(isset($results->issues)) {
            $storypoints=0;
            $storycount=0;
            $bugcount=0;
//            dd($results);
            foreach($results->issues as $issue) {
                if("Story" == $issue->fields->issuetype->name) {
                    $storycount++;
                    if ($issue->fields->customfield_10105 > 0) {
                        $storypoints+=$issue->fields->customfield_10105;
                    }
                }
                else {
                    $bugcount++;
                }
            }
//            dd(count($results->issues));
            return array(count($results->issues),$storypoints,$storycount,$bugcount);
        } else {
            return array();
        }
    }
        
}