<?php
namespace App\Http\Controllers;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cache;

define('EXPIRY', 5); //how many minutes?)
define('DEBUG', FALSE); //how many minutes?)
date_default_timezone_set('America/Los_Angeles');

class VX extends BaseController
{
    //send test or final email
    public function sendVXEmail($type="test"){
        $project = "vx";
        $template = "vx";
        $data = array(
            'project' => $project,
            'launched' => $this->getLaunchedProjects($project),
            'active' => $this->getActiveProjects($project),
            'results' => $this->getProjectsWithRecentResults($project),
            'contacts' => Config::get($project .'.contacts'),
        );

        //send email
        if ($type == "test") {
            //TEST EMAIL
            $toEmails = array('vx-tpm@justin.tv','asclarkm@justin.tv');
            \Illuminate\Support\Facades\Mail::send($template, $data, function ($message) use ($project, $toEmails) {
                $emailSubject = '[TEST] VX Bi-Weekly Program Update: ' . date('M j', strtotime('-2 weeks')) . ' - ' . date('M j');
                $message->to($toEmails)->subject($emailSubject);
                $message->from(env('MAIL_FROM'));
            });
        } elseif ($type == "final") {
//            //FINAL EMAIL
            $toEmails = array('vx-program-update@justin.tv');
            \Illuminate\Support\Facades\Mail::send($template, $data, function ($message) use ($project, $toEmails) {
                $emailSubject = 'VX Bi-Weekly Program Update: ' . date('M j', strtotime('-2 weeks')) . ' - ' . date('M j');
                $message->to($toEmails)->subject($emailSubject);
                $message->from(env('MAIL_FROM'));
            });
        }
        return View::make($template, $data);
    }

    //create VX update
    public function showVXItem($project = "vx")
    {
        $template = "vx";
        return View::make($template,
            array(
                'project' => $project,
                'launched' => $this->getLaunchedProjects($project),
                'active' => $this->getActiveProjects($project),
                'results' => $this->getProjectsWithRecentResults($project),
                'contacts' => Config::get($project .'.contacts'),
            )
        );
    }

    //jira rest call for projects in active development
    public function getActiveProjects($project = "vx")
    {
        $jql = Config::get($project .'.jira.vxActiveJQL');
        $consumerKey = Config::get($project .'.jira.consumerKey');
        $oauthToken = Config::get($project . '.jira.oauthToken');
        $oauthTokenSecret = Config::get($project . '.jira.oauthTokenSecret');
        $baseURI = Config::get($project . '.jira.baseURI');

        $today = date("Ymd");
        $key = "active_" . $today . "_" . $project;

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(30);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $jql));
            Cache::add($key, $results, $expiresAt);
        }

        if(isset($results->issues)) {
            usort($results->issues, [__CLASS__, 'customStatusOrder']);
            return $results->issues;
        } else {
            return array();
        }
    }

    // Needed since our status ordering request was denied: https://jira.twitch.com/browse/JAR-2368
    private static function customStatusOrder($issue1, $issue2)
    {
        $statusOrdering = array(0 => 'Spec In Progress', 1 => 'Design In Progress', 2 => 'In Development', 3 => 'Live/Final Analysis');
        $statusOrdinal1 = array_search($issue1->fields->status->name, $statusOrdering);
        $statusOrdinal2 = array_search($issue2->fields->status->name, $statusOrdering);
        if ($statusOrdinal1 != $statusOrdinal2) {
            return $statusOrdinal1 - $statusOrdinal2;
        }
        else {
            $dueDate1 = $issue1->fields->duedate ?: '2100-01-01';
            $dueDate2 = $issue2->fields->duedate ?: '2100-01-01';

            return $dueDate1 < $dueDate2 ? -1 : 1;
        }
    }

    //jira rest call for projects launched
    public function getLaunchedProjects($project = "vx")
    {
        $jql = Config::get($project .'.jira.vxLaunchedJQL');
        $consumerKey = Config::get($project .'.jira.consumerKey');
        $oauthToken = Config::get($project . '.jira.oauthToken');
        $oauthTokenSecret = Config::get($project . '.jira.oauthTokenSecret');
        $baseURI = Config::get($project . '.jira.baseURI');

        $today = date("Ymd");
        $key = "launched_" . $today . "_" . $project;

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(30);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $jql));
            Cache::add($key, $results, $expiresAt);
        }

        if(isset($results->issues)) {
            return $results->issues;
        } else {
            return array();
        }
    }

    //jira rest call for projects with results, filtered to recent
    public function getProjectsWithRecentResults($project = "vx")
    {
        $jql = Config::get($project .'.jira.vxHasProductResultsJQL');
        $consumerKey = Config::get($project .'.jira.consumerKey');
        $oauthToken = Config::get($project . '.jira.oauthToken');
        $oauthTokenSecret = Config::get($project . '.jira.oauthTokenSecret');
        $baseURI = Config::get($project . '.jira.baseURI');

        $today = date("Ymd");
        $key = "results_" . $today . "_" . $project;

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(30);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $jql));
            Cache::add($key, $results, $expiresAt);
        }

        if(isset($results->issues)) {
            return array_filter($results->issues, [__CLASS__, 'recentResults']);
        } else {
            return array();
        }
    }

    private static function recentResults($issue)
    {
        return self::getTimeSinceLastChange($issue->fields->customfield_20800, $issue->changelog, "Product Results") <= 14;
    }

    public static function getShortName($fullName){
        $array = explode(" ", $fullName . "");
        if(count($array) >= 2 && $array[1][0]){
            return $array[0] . " " . $array[1][0];
        }
        return $fullName;
    }

    //time since last change of a given field
    public static function getTimeSinceLastChange($issueAge, $changelog, $fieldName)
    {
        if($changelog->total > 0)
        {
            $indx = count($changelog->histories);
            while($indx)
            {
                $history = $changelog->histories[--$indx];
                if($history->items[0]->field == $fieldName) {
                    $diff = floor((time() - strtotime($history->created)) / 86400);  // (60 * 60 * 24)
                    return $diff;
                }
            }
        }
        return $issueAge; //return age of ticket if field has never been updated
    }
}