<?php
namespace App\Http\Controllers;

use Carbon\Carbon;
use Google_Service_Sheets;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cache;
use GuzzleHttp\Client;

define('EXPIRY', 10); //how many minutes?)
define('DEBUG', TRUE); //how many minutes?)
date_default_timezone_set('America/Los_Angeles');
ini_set("auto_detect_line_endings", "1");

//Look back at launched items for the current year
class WOR extends BaseController
{

    var $dates; //week, start date, end date of last full week

    //list all projects with WOR config setup
    public function showAll() {
        $allConfigs = Config::all();
//        dd($allConfigs);
        $template = "wordirectory";
        return View::make($template, array(
            "allConfigs" => $allConfigs
        ));
    }

    //default to 'ce' if no key set
    //which team = jira project to filter to
    public function showPreview($project = "ce")
    {
        $template = "worpreview"; //blade template
        $templateData = $this->makeTemplateData($project);
        return View::make($template, $templateData);
    }

    //generate $templateData for HTML page and email
    public function makeTemplateData($project = "ce") {
        $this->dates = $this->getLastFullWeekStartEnd();
        $templateData = array(
            'project' => $project,
            'date' => $this->dates,
            'projectName' => Config::get($project . ".wor.businessUnit") . " Weekly Operational Review",
            'availabilityPerService' => $this->getAvailabilityPerService($project, $this->dates["start"]->timestamp, $this->dates["end"]->timestamp),
            'overallServiceAvailability' => $this->getOverallAvailability($project, $this->dates["start"]->timestamp, $this->dates["end"]->timestamp),
            'availabilityExpectations' => $this->getAvailabilityExpectations($project),
            'overallAvailabilityExpectations' => $this->getOverallAvailabilityExpectations($project),
            'overallAvailabilityTrend' => $this->getAvailabilityMonthlyTrend($project),
            'securityExpectations' => $this->getSecurityExpectations($project),
            'AWSCosts' => $this->getAWSCosts($project),
            'CWT' => $this->getCWT($project, $this->dates["start"]->year, $this->dates["start"]->weekOfYear),
            'unclassifiedJira' => $this->getJiraResults($project, "unclassifiedJira"),
            'sev3PlusJira' => $this->getJiraResults($project, "sev3PlusJira"),
            'ooslaJira' => $this->getJiraResults($project, "ooslaJira"),
            'availabilityRemediation' => $this->getJiraResults($project, "availabilityRemediation"),
            'securityVulnerabilities' => $this->getJiraResults($project, "securityVulnerabilities"),
            'securityRemediation' => $this->getJiraResults($project, "securityRemediation"),
            'incidentsOwned' => $this->getJiraResults($project, "incidentsOwned"),
            'contacts' => Config::get($project . '.contacts'),
        );
        return $templateData;
    }

    //sends out WOR as email
    public function sendEmail($project = "ce", $type = "test"){
        $template = "worpreview"; //blade template
        $data = $this->makeTemplateData("$project");
        //send email
        if ($type == "test") {
            //TEST EMAIL
            $toEmails = explode(',', env('MAIL_TO_TEST_WOR'));
            \Illuminate\Support\Facades\Mail::send($template, $data, function ($message) use ($data, $toEmails) {
                $emailSubject = env('MAIL_SUBJECT_PREPEND_TEST') . $data['projectName'] . " (" . $data["date"]["start"]->format('M d') . " - " . $data["date"]["end"]->format('M d') . ")";
                $message->to($toEmails)->subject($emailSubject);
                $message->from(env('MAIL_FROM'));
            });
        } elseif ($type == "final") {
            //FINAL EMAIL
            $toEmails = explode(',', env('MAIL_TO_WOR'));
            \Illuminate\Support\Facades\Mail::send($template, $data, function ($message) use ($data, $toEmails) {
                $emailSubject = $data['projectName'] . " (" . $data["date"]["start"]->format('M d') . " - " . $data["date"]["end"]->format('M d') . ")";
                $message->to($toEmails)->subject($emailSubject);
                $message->from(env('MAIL_FROM'));
            });
        }

        //show wpu on browser after it has been emailed out
        return View::make($template, $data);
    }

    //weeks are Monday to Sunday according
    public function getLastFullWeekStartEnd() {
        return ["week" => Carbon::today()->subWeek(1)->weekOfYear,
            "start" => Carbon::today()->subWeek(1)->startOfWeek(),
            "end" => Carbon::today()->subWeek(1)->endOfWeek()];
    }

    //start + end in epoch time
    public function getAvailabilityPerService($project = "ce", $start = 1552331606, $end = 1555701326)
    {
        if (Config::get($project . '.wor.availabilityPerService') != null && Config::get($project . '.wor.availabilityPerService.restUrl')!= null) {
            $url = Config::get($project . '.wor.availabilityPerService.restUrl') . "&from=" . $start . "&until=" . $end;
//            dd($url);
            $client = new Client();
            $res = $client->get($url);
//            dd($res->getStatusCode()); //check response

            $today = date("Ymd");
            $modelReflector = new \ReflectionClass(__CLASS__);
            $method = $modelReflector->getMethod(__FUNCTION__)->getName();
            $key = md5($method . $project . $start . $end . $today);

            if (Cache::has($key) && DEBUG) {
                $response = Cache::get($key);
            } else {
                $expiresAt = \Carbon\Carbon::now()->addMinutes(EXPIRY);
                $response = json_decode($res->getBody());
                Cache::add($key, $response, $expiresAt);
            }
//            dd($response);

            $averages = array();

            foreach ($response as $service) {
                $serviceName = $service->target;
                $total = 0;
                $denominator = 0;
                $numerator = array();
                foreach ($service->datapoints as $value) {
                    if($value[0] != null) {
                        $total += $value[0];
                        $numerator[] = $value[0];
                        $denominator++;
                    }
                }
                if ($denominator != 0) {
                    $averages[$serviceName] = round($total / $denominator, 2);
                }
//                $averages[$serviceName] = round($total / count($service->datapoints), 2); //use nulls
            }
            ksort($averages);
//            dd($averages);
            return $averages;
        } else {
            return array();
        }
    }


    //take the mean of all the services, start + end in epoch time
    public function getOverallAvailability($project, $start = 1552331606, $end = 1555701326)
    {
        if (Config::get($project . '.wor.availabilityPerService') != null && Config::get($project . '.wor.availabilityPerService.restUrl')!= null) {
            return round(\Helpers::mmmr($this->getAvailabilityPerService($project, $start, $end), "mean"), 2);
        } else {
            return null;
        }
    }

    //pull from google sheets https://docs.google.com/spreadsheets/d/1B_jYTcxDmZQ_tkSITTZMLx07Nw_SNsoiXswpB4WicXk/edit#gid=598416105
    public function getAvailabilityExpectations($project)
    {
        $sheetID = Config::get('wor.availabilityExpectations.sheetId');
        $sheetRange = Config::get($project . '.wor.availabilityExpectations.sheetRange');
        $today = date("Ymd");
        $modelReflector = new \ReflectionClass(__CLASS__);
        $method = $modelReflector->getMethod(__FUNCTION__)->getName();
        $key = md5($method . $project . $today);

        if ($sheetRange == null) {
            return array();
        }

        if (Cache::has($key) && DEBUG) {
            $sheetData = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(EXPIRY);
            $sheetData = \GoogleSheets::getRangeData($sheetID, $sheetRange);
            Cache::add($key, $sheetData, $expiresAt);
        }

//        dd($sheetData); //find out what column the data is on
//        0 => "T-Team"
//        1 => "ET-Team Lead"
//        2 => "Manager"
//        3 => "Primary Owner"
//        4 => "Service ID"
//        5 => "Service"
//        6 => "01. Postmortems"
//        7 => "02. Correction of Errors"
//        8 => "03. Runbooks"
//        9 => "04. Deployment runbook"
//        10 => "05. Data Recovery Plan"
//        11 => "06. Dashboard"
//        12 => "07. Alerting"
//        13 => "08. Service SLAs"
//        14 => "09. Per-endpoint SLAs"
//        15 => "10. Test Automation"
//        16 => "11. Testing Continuous Improvement"
//        17 => "12. Unit Testing"
//        18 => "13. Integration Testing"
//        19 => "14. Pre-build Testing"
//        20 => "15. Operational Review"
//        21 => "16. Track ownership"
//        22 => "17. AWS Accounts"
//        23 => "18. Bug Triage and Remediation"
//        24 => "19. Change management"
//        25 => "20. Canary environments"
//        26 => "21. Code Review"
//        27 => "22. Source Control"
//        28 => "23. On-Call"
//        29 => "24. Architectural Review"
//        30 => "25. Architecture Documentation"
//        31 => "26. Data Backup Auditing"
//        32 => "27. Log Rotation"
//        33 => "28. Off-box Logging"
//        34 => "Notes"
//        35 => "Include in Availability Calc?"
//        36 => "Status-Done"
//        37 => "Status-2018"
//        38 => "Status-2019"
//        39 => "Status-2020"
//        40 => "Status-No Plan"
//        41 => "Status-NA"
//        42 => "Status-Won't Do"
//        43 => "Status-Fulton 2019"
//        44 => "Status-Fulton 2020"
//        45 => "Total"
        $srvcName = 5; //what index
        $srvcStatusDone = 36;
        $srvcStatus2018 = 37;
        $srvcStatus2019 = 38;
        $srvcStatus2020 = 39;
        $srvcStatusNoPlan = 40;
        $srvcWontDo = 42;

        $average = array();
        for ($i = 1; $i < count($sheetData); $i++) {
            if (isset($sheetData[$i][$srvcStatusDone]) && (int)$sheetData[$i][$srvcStatusDone]+(int)$sheetData[$i][$srvcStatus2018]+(int)$sheetData[$i][$srvcStatus2019]+(int)$sheetData[$i][$srvcStatus2020]+(int)$sheetData[$i][$srvcStatusNoPlan]+(int)$sheetData[$i][41] > 0) {
                $average[$sheetData[$i][$srvcName]]["status"] = (int)$sheetData[$i][$srvcStatusDone] + (int)$sheetData[$i][$srvcStatus2018] + (int)$sheetData[$i][$srvcStatus2019];
                $average[$sheetData[$i][$srvcName]]["target"] = ceil(($average[$sheetData[$i][$srvcName]]["status"] / ($average[$sheetData[$i][$srvcName]]["status"] + (int)$sheetData[$i][$srvcStatus2020] + (int)$sheetData[$i][$srvcStatusNoPlan]+(int)$sheetData[$i][$srvcWontDo])) * 100);
            }
        }

        ksort($average);
//        dd($average);
        return $average;
    }

    //pull from google sheets https://docs.google.com/spreadsheets/d/1Q0iwm4JgG9AOgl9_TkK7YgDmzFcCcbXhppOJp_aCmg8/edit#gid=108601309
//    0 => ""
//    1 => ""
//    2 => "Month"
//    3 => "Jan"
//    4 => "Feb"
//    5 => "Mar"
//    6 => "Apr"
//    7 => "May"
//    8 => "Jun"
//    9 => "Jul"
//    10 => "Aug"
//    11 => "Sep"
//    12 => "Oct"
//    13 => "Nov"
//    14 => "Dec"
//    15 => ""
//    16 => "YTD"
    public function getAvailabilityMonthlyTrend($project)
    {
        if(Config::get("$project.wor.availabilityExpectations.KPIsheetSTL") != null) {
            $sheetID = Config::get('wor.availabilityExpectations.KPIsheetId');
            $sheetRange = Config::get('wor.availabilityExpectations.KPIsheetRange');
            $today = date("Ymd");
            $modelReflector = new \ReflectionClass(__CLASS__);
            $method = $modelReflector->getMethod(__FUNCTION__)->getName();
            $key = md5($method . $project . $today);

            if (Cache::has($key) && DEBUG) {
                $sheetData = Cache::get($key);
            } else {
                $expiresAt = \Carbon\Carbon::now()->addMinutes(EXPIRY);
                $sheetData = \GoogleSheets::getRangeData($sheetID, $sheetRange);
                Cache::add($key, $sheetData, $expiresAt);
            }
            $stl = array_column($sheetData, 1);
            $index = array_search(Config::get("$project.wor.availabilityExpectations.KPIsheetSTL"), $stl); //found STL
            $sheetDataOriginalSize = count($sheetData);

            //remove unneeded rows
            for ($i = 0; $i < $index; $i++) {
                unset($sheetData[$i]);
            }
            for ($i = $i + 6; $i < $sheetDataOriginalSize; $i++) {
                unset($sheetData[$i]);
            }

            //initialize months
            $months[] = null;
            for ($i = 1; $i <= 12; $i++) {
                $months[] = Carbon::create(2019, $i, 1)->format("Y-m");
            }
            $months[] = null;
            $months[] = "YTD";


            $data = array();
            for ($i = $index + 1; $i < $index + count($sheetData); $i++) {
                for ($j = 2; $j < count($sheetData[$index + 1]); $j++) {
                    $key = trim($sheetData[$i][1]);
                    $time = $months[$j - 2];
                    $data[$time][$key] = $sheetData[$i][$j];
                }
            }

            reset($data);
            unset($data[key($data)]);

            foreach ($data as $key => $metrics) {
                if ($key != "YTD") {
                    $date = Carbon::createFromFormat("Y-m", $key);
                    if ($date->greaterThan(Carbon::now()) || $date->lessThan(Carbon::now()->subMonths(3))) {
                        unset($data[$key]);
                    }
                }
            }
//        dd($data);
            return ($data);
        } else {
            return array();
        }
    }


    //take the mean of all services, should it ignore the new stuff?
    public function getOverallAvailabilityExpectations($project) {
        $status = array();
        $target = array();
        foreach($this->getAvailabilityExpectations($project) as $key => $value) {
            $status[] = $value['status'];
            $target[] = $value['target'];
        }
        if (count($status) > 0) {
            return array("status" => \Helpers::mmmr($status), "target" => \Helpers::mmmr($target));
        } else {
            return array("status" => 0, "target" => 0);
        }
    }

    //pull from google sheets https://docs.google.com/spreadsheets/d/1OAO0xceITLdLtlav2HoxdHXOIxkFwXB3A1-k6kE22qo/edit#gid=893590975
    public function getSecurityExpectations($project)
    {
        $sheetID = Config::get('wor.securityExpectations.sheetId');
        $sheetRange = Config::get($project . '.wor.securityExpectations.sheetRange');

        $today = date("Ymd");
        $modelReflector = new \ReflectionClass(__CLASS__);
        $method = $modelReflector->getMethod(__FUNCTION__)->getName();
        $key = md5($method . $project . $today);

        if (Cache::has($key) && DEBUG) {
            $sheetData = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(EXPIRY);
            $sheetData = \GoogleSheets::getRangeData($sheetID, $sheetRange);
            Cache::add($key, $sheetData, $expiresAt);
        }

//        dd($sheetData); //find out what column the data is on
        //Row 1 = Headers
        //Row 5 = Data
        $values = array(
            $sheetData[1][0] . " for " . $sheetData[0][0] => $sheetData[5][0],
            $sheetData[1][1] => $sheetData[5][1],
            $sheetData[1][2] => $sheetData[5][2],
            $sheetData[1][3] => $sheetData[5][3],
            $sheetData[1][4] => $sheetData[5][4]
        );
//        dd($values);
        return $values;
    }

    //pull from google sheets https://docs.google.com/spreadsheets/d/1-1JvB4k8s3IK4lvfrFSpbZb7tNGOV6H2LGbjYsfxoxY/edit#gid=1957448860
    public function getAWSCosts($project)
    {
        $sheetID = Config::get('wor.AWSCosts.sheetId');
        $sheetRange = Config::get($project . '.wor.AWSCosts.sheetRange');


        $today = date("Ymd");
        $modelReflector = new \ReflectionClass(__CLASS__);
        $method = $modelReflector->getMethod(__FUNCTION__)->getName();
        $key = md5($method . $project . $today);

        if (Cache::has($key) && DEBUG) {
            $sheetData = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(EXPIRY);
            $sheetData = \GoogleSheets::getRangeData($sheetID, $sheetRange);
            Cache::add($key, $sheetData, $expiresAt);
        }

//        Sheet Format https://docs.google.com/spreadsheets/d/15j3X_rfyObZ1DSKhVELnlIZriIYb6c080MpknMtZUxQ/edit#gid=267322130
//        Date = 0
//        Total Cost = 2
//        Goal = 3
//        vs Goal ($) = 4
//        vs Goal (%) = 5
//        % MoM = 6
//        Utilization Opportunity = 9
//        % MoM = 10
//        YTD ($) = 11
//        $/kHW = 13
//        Goal = 14
//        vs Goal ($) = 15
//        vs Goal (%) = 16
//        % MoM = 17
//        Utilization Score = 19
//        Goal = 20
//        vs Goal (bsp) = 21
//        MoM (bsp) = 22

//        dd($sheetData);

        $aws = array();
        for($i = 0; $i < count($sheetData[0]); $i++) {
            if (Carbon::parse($sheetData[0][$i])->isFuture() != 1) {
                $key = Carbon::parse($sheetData[0][$i])->format("Y-m");

                if (!isset($sheetData[2][$i]) || $sheetData[2][$i] == "") {
                    $aws[$key]["Cost"] = "TBD";
                } else {
                    $aws[$key]["Cost"] = $sheetData[2][$i];
                }

                $aws[$key]["Cost Goal"] = $sheetData[3][$i];
                $aws[$key]["vs Cost Goal %"] = $sheetData[5][$i];
                $aws[$key]["Cost MoM %"] = $sheetData[6][$i];
                $aws[$key]["Utilization Opportunity"] = $sheetData[9][$i];
                $aws[$key]["$/kHW"] = $sheetData[13][$i];
                $aws[$key]["$/kHW Goal"] = $sheetData[14][$i];

                if (!isset($sheetData[19][$i]) || $sheetData[19][$i] == "") {
                    $aws[$key]["Utilization Score"] = "TBD";
                } else {
                    $aws[$key]["Utilization Score"] = $sheetData[19][$i];
                }

                if (!isset($sheetData[20][$i]) || $sheetData[20][$i] == "") {
                    $aws[$key]["Utilization Goal"] = "TBD";
                } else {
                    $aws[$key]["Utilization Goal"] = $sheetData[20][$i];
                }
            }
        }
//        dd($aws);
        return $aws; //find out what column the data is on
    }

    //pull data from latency API
    public function getCWT($project = "ce", $year = 2019, $week = 17)
    {
        if(Config::get($project . '.wor.CWT') != null) {
            $client = new Client(['http_errors' => false]); //guzzlehttp client
            $url = Config::get($project . '.wor.CWT.restUrl');
            $url = preg_replace('/\[\[YEAR\]\]/m', $year, $url);
            $url = preg_replace('/\[\[WEEK\]\]/m', "w" . $week, $url);
            $headers = ["Authorization" => "Bearer " . env("LATENCY_JWT")];
            $response = $client->request('GET', $url, [
                'headers' => $headers
            ]);

            if ($response->getStatusCode() == 401) {
                return array();
            }

            $today = date("Ymd");
            $modelReflector = new \ReflectionClass(__CLASS__);
            $method = $modelReflector->getMethod(__FUNCTION__)->getName();
            $key = md5($method . $project . $year . $week . $today);

            if (Cache::has($key) && DEBUG) {
                $rows = Cache::get($key);
            } else {
                $expiresAt = \Carbon\Carbon::now()->addMinutes(EXPIRY);
                $rows = str_getcsv($response->getBody()->getContents(), "\n");
                Cache::add($key, $rows, $expiresAt);
            }

//            dd($rows); //debugging raw data
            $header = str_getcsv(array_shift($rows));
            $csv = array();
            foreach ($rows as $row) {
                $csv[] = array_combine($header, str_getcsv($row));
            }

            $csv = array_filter($csv,
                function ($val, $key) use ($project) { // N.b. $val, $key not $key, $val
                    return $val["Org Leader"] == Config::get($project . ".wor.CWT.orgLeader");
                }, ARRAY_FILTER_USE_BOTH);
            return $csv;
        } else {
            return null;
        }
    }


    //pull from google sheets https://docs.google.com/spreadsheets/d/1HAhxEY-ieS-qLLgA3voGQRqPFIBAa05hx8b3ouaNpA4/edit#gid=0
    public function writeHistoricalDataFromSheet($project = "ce", $writeType = "test")
    {
        $this->dates = $this->getLastFullWeekStartEnd();
        $sheetID = Config::get($project . '.wor.trackingSheet.sheetId');
        $sheetRange = Config::get($project . '.wor.trackingSheet.sheetRange');

        $today = date("Ymd");
        $modelReflector = new \ReflectionClass(__CLASS__);
        $method = $modelReflector->getMethod(__FUNCTION__)->getName();
        $key = md5($method . $project . $today);

        if (Cache::has($key) && DEBUG) {
            $sheetData = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(EXPIRY);
            $sheetData = \GoogleSheets::getRangeData($sheetID, $sheetRange);
            Cache::add($key, $sheetData, $expiresAt);
        }

        $currentWeeksMonday = Carbon::today()->subWeek(1)->startOfWeek();
//        dd($sheetData);

        $columnToWriteData = 0;
        for ($i = 2; $i < count($sheetData[1]); $i++) {
            if ($currentWeeksMonday->equalTo(Carbon::parse($sheetData[1][$i])) == TRUE) {
                $columnToWriteData = \GoogleSheets::getA1Notation($i);
                $previousWeekA1 = \GoogleSheets::getA1Notation($i-1);
                break;
            }
        }

        //for week over week spreadsheet initialize values
        $jiraTypes = ["securityVulnerabilities", "securityRemediation"];
        $i = 4; //row number to start writing on google sheet
        $values = array();
        $headers = array();

        //google sheet headers
        $headers[] = "Security Vulnerabilities";
        $headers[] = "Goal";
        $headers[] = "WOW %";
        $headers[] = "Open Sec Remediation Items";
        $headers[] = "Goal";
        $headers[] = "WOW %";
        $headers[] = "Incidents Owned";
        $headers[] = "WOW %";
        $headers[] = "Service Availability";
        $headers[] = "Goal";
        $headers[] = "WOW %";
        $headers[] = "Average Time to Close COE";
        $headers[] = "Goal";
        $headers[] = "WOW %";
        $headers[] = "Open Availability Remediation Items";
        $headers[] = "Goal";
        $headers[] = "WOW %";
        $headers[] = "Issues OOSLA";
        $headers[] = "Goal";
        $headers[] = "WOW %";
        $headers[] = "Unclassified Issues";
        $headers[] = "Goal";
        $headers[] = "WOW %";
        $headers[] = "Open Sev-3+ Issues";
        $headers[] = "Goal";
        $headers[] = "WOW %";
        $headers[] = "AWS Spend";
        $headers[] = "Goal";
        $headers[] = "WOW %";
        $headers[] = "Percent Sec. Expectations Met";
        $headers[] = "Goal";
        $headers[] = "WOW %";
        $headers[] = "Percent Avail. Expectations Met";
        $headers[] = "Goal";
        $headers[] = "WOW %";

        //week over week formula to use in Google Sheet using anonymous function
        $wowString =  function () use (&$i, &$previousWeekA1, &$columnToWriteData) {
            return "=IFS(ISBLANK($previousWeekA1$i), \"\", $columnToWriteData$i>0, ($columnToWriteData$i-$previousWeekA1$i)/$columnToWriteData$i, $previousWeekA1$i>0, ($columnToWriteData$i-$previousWeekA1$i)/$previousWeekA1$i,  $columnToWriteData$i=0, 0)";
        };

        foreach ($jiraTypes as $jiraType) {
            $values[][] = count($this->getJiraResults($project, $jiraType));
            $values[][] = 0; //goal should be 0
            $values[][] = $wowString();
            $i = $i + 3;
        }

        $jiraTypes = ["incidentsOwned"];
        foreach ($jiraTypes as $jiraType) {
            $values[][] = count($this->getJiraResults($project, $jiraType));
            $values[][] = $wowString();
            $i = $i + 2;
        }

        //Service Availability
        $values[][] = $this->getOverallAvailability($project, $this->dates["start"]->timestamp, $this->dates["end"]->timestamp) . "%";
        $values[][] = Config::get($project . ".wor.availabilityPerService.target");
        $values[][] = $wowString();
        $i = $i + 3;

        //Average Time to Close COE TODO
        $values[][] = null;
        $values[][] = null;
        $values[][] = $wowString();
        $i = $i + 3;


        $jiraTypes = ["availabilityRemediation", "ooslaJira", "unclassifiedJira", "sev3PlusJira"];
        foreach ($jiraTypes as $jiraType) {
            $values[][] = count($this->getJiraResults($project, $jiraType));
            $values[][] = 0; //goal should be 0
            $values[][] = $wowString();
            $i = $i + 3;
        }

        //AWS Spend
//        dd($this->getAWSCosts($project));
        $AWSCosts = $this->getAWSCosts($project)[Carbon::today()->subWeek(1)->startOfMonth()->format("Y-m")];
        $values[][] = $AWSCosts['Cost'];
        $values[][] = $AWSCosts['Cost Goal'];
        $values[][] = $wowString();
        $i = $i + 3;

        //Percent Sec. Expectations Met
//        dd($this->getSecurityExpectations($project));
        $values[][] = $this->getSecurityExpectations($project)["Last Month"];
        $values[][] = $this->getSecurityExpectations($project)["2019 Goals"];
        $values[][] = $wowString();
        $i = $i + 3;

        //Percent Avail. Expectations Met
        $availabilityExpectations = $this->getAvailabilityMonthlyTrend($project)[Carbon::today()->subWeek(1)->startOfMonth()->format("Y-m")];
        $values[][] = $availabilityExpectations["% Complete"];
        $values[][] = $availabilityExpectations["2019 Goal"];
        $values[][] = $wowString();
        $i = $i + 3;

        //Customer Wait Time - Dashboard Live (ms)
        $cwt = $this->getCWT($project, $this->dates["start"]->year, $this->dates["start"]->weekOfYear);
//        dd($cwt);
//        dd($values);
        if(!(is_null($cwt))) {
            foreach ($cwt as $metric) {
                $values[][] = $metric["Current"];
                $values[][] = $metric["Phased Goal"];
                $values[][] = $wowString();
                $headers[] = "CWT - " . $metric["Experience"] . " - " . $metric["Metric"];
                $headers[] = "Goal";
                $headers[] = "WOW %";
                $i = $i + 3;
            }
        }
        

        if ($writeType == "final") {
            $body = new \Google_Service_Sheets_ValueRange([
                'values' => $values
            ]);
            $params = [
                'valueInputOption' => "USER_ENTERED"
            ];

            $writeRange = "Sheet1!" . $columnToWriteData . "4";
            putenv('GOOGLE_APPLICATION_CREDENTIALS=../../google-7ed5d6ebdcae.json');
            $client = new \Google_Client();
            $client->useApplicationDefaultCredentials();
            $client->setScopes(Google_Service_Sheets::SPREADSHEETS);
            $service = new Google_Service_Sheets($client);
            $result = $service->spreadsheets_values->update($sheetID, $writeRange,
                $body, $params);
            $count = $result->getUpdatedCells();
        } else {
            $count = 0;
        }

        $this->dates = $this->getLastFullWeekStartEnd();
        $template = "worsnapshot"; //blade template


        return View::make($template, array(
            'project' => $project,
            'date' => $this->dates,
            'projectName' => Config::get($project . ".wor.businessUnit") . " Weekly Operational Review",
            "numberOfCellsUpdated" => $count,
            "headers" => $headers,
            "sheetValues" => $values
        ));
    }

    //generic jira GET method using configs
    public function getJiraResults($project = "ce", $type = "unclassifiedJira")
    {
        $jql = Config::get('wor.' . $type .'.jql');
        $jql = preg_replace('/\[\[PROJECT\]\]/m',  Config::get($project . '.wor.projects'), $jql); //replace [[PROJECT]] with actual projects
        $jql = preg_replace('/\[\[TEAM\]\]/m',  Config::get($project . '.wor.team'), $jql); //replace [[PROJECT]] with actual projects
        $jql = preg_replace('/\[\[SERVICESIMPACTED\]\]/m',  Config::get($project . '.wor.servicesImpacted'), $jql); //replace [[SERVICESIMPACTED]] with field values
        $jqlFragment = Config::get('wor.jqlFragment');
        $jql = preg_replace('/\[\[JQL\]\]/m', addslashes($jql), $jqlFragment); //form full query for JIRA REST API

        $consumerKey = Config::get('wor.consumerKey');
        $oauthToken = env("JIRA_OAUTH_TOKEN");
        $oauthTokenSecret = env("JIRA_OAUTH_SECRET");
        $baseURI = Config::get( 'wor.baseURI');

        $today = date("Ymd");
        $modelReflector = new \ReflectionClass(__CLASS__);
        $method = $modelReflector->getMethod(__FUNCTION__)->getName();
        $key = md5($method . $project . $type . $today);

        if (Cache::has($key) && DEBUG) {
            $results = Cache::get($key);
        } else {
            $expiresAt = \Carbon\Carbon::now()->addMinutes(EXPIRY);
            $jira = new \Jira();
            $results = json_decode($jira->post($oauthToken, $oauthTokenSecret, $consumerKey, $baseURI, "search", $jql));
            Cache::add($key, $results, $expiresAt);
        }

        if(isset($results->issues)) {
            return $results->issues;
        } else {
            return array();
        }
    }
}