#!/bin/bash
# This bash script creates a vpc for your account
set -euo pipefail

STACK_NAME=vpc
STACK_FILE=file://3-subnets/vpc.yaml

function cli_works() {
	aws sts get-caller-identity &> /dev/null
}

# Returns non zero if the stack already exists
function stack_exists() {
	aws cloudformation describe-stacks --stack-name ${STACK_NAME} &> /dev/null
}

function public_subnets() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='AllPublicSubnets'].OutputValue" --output text
}

function private_subnets() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='AllPrivateSubnets'].OutputValue" --output text
}

function account_id() {
    aws sts get-caller-identity --query Account --output text
}

function vpc() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='VPC'].OutputValue" --output text
}

function security_group() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='SecurityGroup'].OutputValue" --output text
}

echo "Verifying cli works ..."
if ! cli_works ; then
	echo "Unable to execute get-caller-identity"
	echo "Are you sure you setup your cli correctly?"
	echo "You may need to set the AWS_PROFILE variable"
	exit 1
fi

echo "Account ID: $(account_id)"
echo "Region:     $(aws configure get region)"

if ! stack_exists ; then
	echo "stack does not already exist.  Creating"
	echo "Creating stack"
	aws cloudformation create-stack --stack-name ${STACK_NAME} --template-body ${STACK_FILE} --capabilities CAPABILITY_NAMED_IAM
	echo "VPC creation can take a while.  Please be patient"
	echo "Waiting for stack to finish creating"
	echo "You can follow along on the UI at https://us-west-2.console.aws.amazon.com/cloudformation/home"
	aws cloudformation wait stack-create-complete --stack-name ${STACK_NAME}
	echo "Things look good!"
else
	echo "Stack already exists. Updating and pulling existing information.  If you get a 'No updates are to be performed' error, ignore it"
	aws cloudformation update-stack --stack-name ${STACK_NAME} --template-body ${STACK_FILE} --capabilities CAPABILITY_NAMED_IAM 2> /tmp/err || true
	cat /tmp/err
	if ! grep -q  'No updates' /tmp/err; then
		aws cloudformation wait stack-update-complete --stack-name ${STACK_NAME}
	fi
fi

echo "here is your VPN information"

echo "VPC:             $(vpc)"
echo "Public Subnets:  $(public_subnets)"
echo "Private Subnets: $(private_subnets)"
echo "Security Group:  $(security_group)"
