#!/bin/bash
# Uses cloudformation to create an LDAP enabled EC2 instance that you can use to tunnel connections to resources inside your account's VPN
set -euo pipefail

export AWS_DEFAULT_REGION=${AWS_DEFAULT_REGION-${AWS_REGION-us-west-2}}
export AWS_REGION=${AWS_DEFAULT_REGION}
export CLUSTER_NAME=${CLUSTER_NAME-}

if [[ -z "${CLUSTER_NAME-}" ]]; then
	echo "Unable to find CLUSTER_NAME env variable.  Set it to the CLUSTER_NAME env variable."
	exit 1
fi

export STACK_NAME=teleport-remote-${CLUSTER_NAME}

# Returns non zero if the stack already exists
function stack_exists() {
	aws cloudformation describe-stacks --stack-name $1 &> /dev/null
}

function sandstorm_role() {
  if [[ ! -z "${SANDSTORM_ROLE-}" ]]; then
    echo ${SANDSTORM_ROLE}
    exit 0
  fi
  # Just a guess ...
  echo "teleport-remote-${CLUSTER_NAME}"
}

function cli_works() {
	aws sts get-caller-identity &> /dev/null
}

function discover_basic_private_subnets() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='AllPrivateSubnets'].OutputValue" --output text
}

function discover_basic_vpc() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='VPC'].OutputValue" --output text
}

function guess_vpc_id() {
  if [[ ! -z "${VPC_ID-}" ]]; then
    echo ${VPC_ID}
    exit 0
  fi
  BASIC_ID=$(discover_basic_vpc)
  if [[ ! -z ${BASIC_ID} ]]; then
    echo ${BASIC_ID}
    return 0
  fi
  aws ec2 describe-vpcs --filters "Name=tag:Name,Values=${ACCOUNT_NAME}" --query 'Vpcs[].VpcId' --output text
}

function guess_private_subnet_id() {
  if [[ ! -z "${SUBNET_ID-}" ]]; then
    echo ${SUBNET_ID}
    exit 0
  fi
  BASIC_ID=$(discover_basic_private_subnets)
  if [[ ! -z ${BASIC_ID} ]]; then
    echo ${BASIC_ID}
    return 0
  fi
  VPC_ID=${1-}
  aws ec2 describe-subnets --query "Subnets[? ! MapPublicIpOnLaunch] | [?VpcId =='$VPC_ID'].SubnetId | [0]" --output text
}

if ! cli_works ; then
	echo "Unable to execute get-caller-identity"
	echo "Are you sure you setup your cli correctly?"
	exit 1
fi

aws sts get-caller-identity

AWS_REGION=${AWS_REGION:-us-west-2}
SCRIPT_VPC_ID=$(guess_vpc_id)
SCRIPT_SUBNET_ID=$(guess_private_subnet_id ${SCRIPT_VPC_ID})
SCRIPT_SANDSTORM_ROLE=$(sandstorm_role)

if stack_exists ${STACK_NAME} ; then
  echo "Your tier already exists!"
  exit 0
fi

if [[ -z "$SCRIPT_VPC_ID" ]]; then
  echo "Unable to guess a VPC_ID for this script.  Pass one with 'env VPC_ID=XYZ'"
  exit 1
fi

if [[ -z "$SCRIPT_SANDSTORM_ROLE" ]]; then
  echo "Unable to detect the sandstorm role.  Pass one with 'env SANDSTORM_ROLE=XYZ'"
  exit 1
fi

if [[ -z "$SCRIPT_SUBNET_ID" ]]; then
  echo "Unable to guess a subnet ID for this script.  Pass one with 'env SUBNET_ID=XYZ'"
  exit 1
fi

aws cloudformation deploy --stack-name ${STACK_NAME} --capabilities CAPABILITY_IAM --template-file security_bastion_cluster.yaml --parameter-overrides ClusterName=${CLUSTER_NAME} Subnets=${SCRIPT_SUBNET_ID} VPC=${SCRIPT_VPC_ID} SandstormRole=${SCRIPT_SANDSTORM_ROLE} PagerDutyIntegrationURL="" AlertsEmail=""

echo "Done!"