#!/bin/bash
# Uses cloudformation to create an LDAP enabled EC2 instance that you can use to tunnel connections to resources inside your account's VPN
set -euo pipefail

export AWS_DEFAULT_REGION=${AWS_DEFAULT_REGION-${AWS_REGION-us-west-2}}
export AWS_REGION=${AWS_DEFAULT_REGION}
export CLUSTER_NAME=${CLUSTER_NAME-}
export LDAP_SERVICE_NAME="com.amazonaws.vpce.us-west-2.vpce-svc-02673a2089b1caf6d"

if [[ -z "${CLUSTER_NAME-}" ]]; then
	echo "Unable to find CLUSTER_NAME env variable.  Set it to the CLUSTER_NAME env variable."
	exit 1
fi

STACK_NAME=ldap-enabled-host
if [[ -z "${ACCOUNT_NAME-}" ]]; then
	echo "Unable to find ACCOUNT_NAME env variable.  Set it to the ACCOUNT_NAME env variable."
	exit 1
fi

# Returns non zero if the stack already exists
function stack_exists() {
	aws cloudformation describe-stacks --stack-name $1 &> /dev/null
}

function cli_works() {
	aws sts get-caller-identity &> /dev/null
}

function discover_basic_private_subnets() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='PrivateSubnetA'].OutputValue" --output text
}

function discover_basic_vpc() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='VPC'].OutputValue" --output text
}

function guess_vpc_id() {
  if [[ ! -z "${VPC_ID-}" ]]; then
    echo ${VPC_ID}
    exit 0
  fi
  BASIC_ID=$(discover_basic_vpc)
  if [[ ! -z ${BASIC_ID} ]]; then
    echo ${BASIC_ID}
    return 0
  fi
  aws ec2 describe-vpcs --filters "Name=tag:Name,Values=${ACCOUNT_NAME}" --query 'Vpcs[].VpcId' --output text
}

function guess_private_subnet_id() {
  if [[ ! -z "${SUBNET_ID-}" ]]; then
    echo ${SUBNET_ID}
    exit 0
  fi
  BASIC_ID=$(discover_basic_private_subnets)
  if [[ ! -z ${BASIC_ID} ]]; then
    echo ${BASIC_ID}
    return 0
  fi
  VPC_ID=${1-}
  aws ec2 describe-subnets --query "Subnets[? ! MapPublicIpOnLaunch] | [?VpcId =='$VPC_ID'].SubnetId | [0]" --output text
}

function guess_ldap_uri() {
  if [[ ! -z "${LDAP_URI-}" ]]; then
    echo ${LDAP_URI}
    exit 0
  fi
  if [[ ! -z "${LDAP_HOST}" ]]; then
    echo "ldaps://${LDAP_HOST}"
    exit 0
  fi
  echo "ldaps://ldap-${ACCOUNT_NAME}.internal.justin.tv"
}

function guess_ldap_group() {
  echo ${LDAP_GROUP-}
}

function instance_id() {
	aws cloudformation describe-stack-resources --stack-name ldap-enabled-host --query 'StackResources[].PhysicalResourceId | [0]' --output=text
}

function private_ip() {
	aws cloudformation describe-stacks --stack-name ${STACK_NAME} --query 'Stacks[].Outputs[].OutputValue | [0]' --output text
}

if ! cli_works ; then
	echo "Unable to execute get-caller-identity"
	echo "Are you sure you setup your cli correctly?"
	exit 1
fi

aws sts get-caller-identity

AWS_REGION=${AWS_REGION:-us-west-2}
SCRIPT_VPC_ID=$(guess_vpc_id)
SCRIPT_SUBNET_ID=$(guess_private_subnet_id ${SCRIPT_VPC_ID})
SCRIPT_LDAP_URI=$(guess_ldap_uri)
LDAP_GROUP=$(guess_ldap_group)

if stack_exists ${STACK_NAME} ; then
  echo "Your bastion host already exists.  Updating stack."
  aws cloudformation update-stack --stack-name ${STACK_NAME} --capabilities CAPABILITY_IAM --template-body file://ldap_host.yaml --parameters ParameterKey=ClusterName,ParameterValue=${CLUSTER_NAME} ParameterKey=LDAPURI,ParameterValue=${SCRIPT_LDAP_URI} ParameterKey=Subnet,ParameterValue=${SCRIPT_SUBNET_ID} ParameterKey=VpcID,ParameterValue=${SCRIPT_VPC_ID} ParameterKey=LDAPGroup,ParameterValue=${LDAP_GROUP} 2> /tmp/err || true
  cat /tmp/err
  if ! grep -q  'No updates' /tmp/err; then
    aws cloudformation wait stack-update-complete --stack-name ${STACK_NAME}
  fi
  echo "  TC=${ACCOUNT_NAME} ssh $(private_ip)"
  exit 0
fi

if [[ -z "$SCRIPT_VPC_ID" ]]; then
  echo "Unable to guess a VPC_ID for this script.  Pass one with 'env VPC_ID=XYZ'"
  exit 1
fi

if [[ -z "$SCRIPT_SUBNET_ID" ]]; then
  echo "Unable to guess a subnet ID for this script.  Pass one with 'env SUBNET_ID=XYZ'"
  exit 1
fi

if [[ -z "$LDAP_GROUP" ]]; then
  echo "Unable to guess an LDAP group for this script.  Pass one with 'env LDAP_GROUP=XYZ'"
  exit 1
fi

aws cloudformation deploy --stack-name ${STACK_NAME} --capabilities CAPABILITY_IAM --template-file ldap_host.yaml --parameter-overrides ClusterName=${CLUSTER_NAME} LDAPURI=${SCRIPT_LDAP_URI} Subnet=${SCRIPT_SUBNET_ID} VpcID=${SCRIPT_VPC_ID} LDAPGroup=${LDAP_GROUP}

echo "Done!"
echo
echo "Find your host here: https://us-west-2.console.aws.amazon.com/ec2/home?region=us-west-2#Instances:search=$(instance_id);sort=instanceId"
echo "The host takes a few minutes to spin up.  Then, check the system log for your instance to see that it connected correctly"
echo "Finally, SSH into your host"
echo "Note: this process usually takes about 5-10 minutes"
echo "  TC=${ACCOUNT_NAME} ssh $(private_ip)"
