#!/bin/bash
# This bash script creates a cloudformation template inside your account to setup the Jenkins user.  It then
# outputs a credentials file you can upload to jenkins so your jenkins jobs can communicate with your AWS account.
set -euo pipefail

STACK_NAME=capacity-planning-readonly

function cli_works() {
	aws sts get-caller-identity &> /dev/null
}

# Returns non zero if the stack already exists
function stack_exists() {
	aws cloudformation describe-stacks --stack-name ${STACK_NAME} &> /dev/null
}

function verify_role() {
	aws iam get-role --role-name twitch-inventory
}

function account_id() {
    aws sts get-caller-identity --query Account --output text
}

if ! cli_works ; then
	echo "Unable to execute get-caller-identity"
	echo "Are you sure you setup your cli correctly?"
	exit 1
fi


echo "Running for account id $(account_id)"

if ! stack_exists ; then
	echo "stack does not already exist.  Creating"
	echo "Creating stack"
	aws cloudformation create-stack --stack-name ${STACK_NAME} --template-body file://capacity-readonly.yaml --capabilities CAPABILITY_NAMED_IAM
	echo "Waiting for stack to finish creating"
	aws cloudformation wait stack-create-complete --stack-name ${STACK_NAME}
else
	echo "Stack already exists. Updating and pulling existing information.  If you get a 'No updates are to be performed' error, ignore it"
	aws cloudformation update-stack --stack-name ${STACK_NAME} --template-body file://capacity-readonly.yaml --capabilities CAPABILITY_NAMED_IAM 2> /tmp/err || true
	cat /tmp/err
	if ! grep -q  'No updates' /tmp/err; then
		aws cloudformation wait stack-update-complete --stack-name ${STACK_NAME}
	fi
fi

echo "Verifying role exists"
verify_role
