package cli

import (
	"bufio"
	"fmt"
	"log"
	"os"
	"syscall"

	"github.com/spf13/cobra"
	"golang.org/x/crypto/ssh/terminal"

	"code.justin.tv/twitch/cli/pkg/config"
	"code.justin.tv/twitch/cli/pkg/ghutil"
)

/*
ConfigureCmd manages a file in ~/.twitch/ which contains credentials to do things like
communicate with Github and Jenkins APIs.
*/
var ConfigureCmd = &cobra.Command{
	Use:   "configure",
	Short: "Creates ~/.twitch/config.yml to hold credentials",
	Run: func(cmd *cobra.Command, args []string) {
		configure(args...)
	},
}

func configure(args ...string) {
	exists, err := config.Exists()
	if err != nil {
		log.Fatal(err)
	}

	var c *config.Config

	if exists {
		c, err = config.Read()
		if err != nil {
			log.Fatal(err)
		}
	} else {
		c = config.New()
	}

	flow(c)

	err = config.Write(c)
	if err != nil {
		log.Fatal(err)
	}
}

func flow(c *config.Config) {
	username := c.Github.Username
	if username == nil {
		user := os.Getenv("USER")
		username = &user
	}

	user, err := prompt("Github Enterprise login", username, false)
	if err != nil {
		log.Fatal(err)
	}

	pw, err := prompt("Github Enterprise password", nil, true)
	if err != nil {
		log.Fatal(err)
	}

	authRequest := &ghutil.GithubAuthorizationRequest{
		Scopes:       []string{"repo", "delete_repo"},
		Note:         "twitch-create-service",
		NoteURL:      "https://git-aws.internal.justin.tv/dta/twitch-create-service",
		ClientID:     "756b1bc9eae27328d2b5",
		ClientSecret: "52f15e486332eed94cf6fa73ce916f3fb9f7433f",
		Fingerprint:  "",
	}

	var token string
	token, err = ghutil.GetCLIToken(user, pw, "", authRequest)
	if err != nil {
		if ghutil.IsOTP(err) {
			otp, err := prompt("Two-factor authentication code", nil, false)
			if err != nil {
				log.Fatal(err)
			}

			token, err = ghutil.GetCLIToken(user, pw, otp, authRequest)
			if err != nil {
				log.Fatal(err)
			}
		} else {
			log.Fatal(err)
		}
	}

	c.Github.Username = &user
	c.Github.Token = &token
}

func prompt(text string, current *string, isPassword bool) (string, error) {
	if isPassword {
		fmt.Print(text + ": ")
		pw, err := terminal.ReadPassword(syscall.Stdin)
		fmt.Println("")
		if err != nil {
			return "", err
		}

		return string(pw), nil
	}

	if current == nil {
		fmt.Printf("%v [None]: ", text)
	} else {
		fmt.Printf("%v [%v]: ", text, *current)
	}

	scanner := bufio.NewScanner(os.Stdin)
	scanner.Scan()

	got := scanner.Text()
	if got == "" {
		return *current, nil
	}
	return got, nil
}
