package cli

import (
	"fmt"
	"log"
	"strings"

	"github.com/google/go-github/github"
	"github.com/spf13/cobra"

	"code.justin.tv/twitch/cli/pkg/config"
	"code.justin.tv/twitch/cli/pkg/ghutil"
)

// SearchCmd represents the search command, which searches GitHub for Twitch CLI plugins
var SearchCmd = &cobra.Command{
	Use:   "search",
	Short: "Search git-aws for twitch-cli subcommands.",
	Run: func(cmd *cobra.Command, args []string) {
		search(args...)
	},
}

type repositorySearchResult struct {
	Name        string
	Description string
	IsFork      bool
}

func search(args ...string) {
	// build client for GitHub
	gitConfig, err := config.Read()
	if err != nil {
		log.Fatal(fmt.Sprintf("Error reading gitConfig: %v", err))
	}
	gitClient, err := ghutil.ClientFromToken(*gitConfig.Github.Token)
	if err != nil {
		log.Fatal(fmt.Sprintf("Error building GitHub client: %v", err))
	}

	// construct search query
	searchTerms := append([]string{`"twitch-cli"`, "fork:false"}, args...)
	query := strings.Join(searchTerms, " ")

	// perform search
	results, err := searchRepositories(gitClient, query)
	if err != nil {
		log.Fatal(fmt.Sprintf("Error searching GitHub: %v", err))
	}

	// display the results
	for _, result := range results {
		s := strings.Split(result.Name, "/")
		_, repo := s[0], s[1]

		// skip repos that don't match the twitch-cli-* format
		if !strings.HasPrefix(repo, "twitch-cli-") {
			continue
		}

		// skip repos with [hide] in the name
		if strings.Contains(strings.ToLower(result.Description), "[hide]") {
			continue
		}

		fmt.Printf("%s", result.Name)
		if result.IsFork {
			fmt.Print(" (fork)")
		}
		fmt.Println()

		if result.Description != "" {
			fmt.Printf("\t%s\n", result.Description)
		}

		fmt.Printf("\tgo get -u code.justin.tv/" + result.Name + "/cmd/...\n")
		fmt.Println()

	}

}

func searchRepositories(client *github.Client, query string) ([]repositorySearchResult, error) {

	opts := &github.SearchOptions{
		Order: "asc",
		ListOptions: github.ListOptions{
			PerPage: 10,
		},
	}
	var repositories []github.Repository
	for {
		searchResult, response, err := client.Search.Repositories(query, opts)
		if err != nil {
			return nil, err
		}
		repositories = append(repositories, searchResult.Repositories...)
		if response.NextPage == 0 {
			break
		}
		opts.ListOptions.Page = response.NextPage
	}

	searchResults := make([]repositorySearchResult, len(repositories))
	for i, repository := range repositories {
		if repository.FullName == nil {
			continue
		}
		searchResults[i].Name = *repository.FullName
		if repository.Description != nil {
			searchResults[i].Description = *repository.Description
		}

		if repository.Fork != nil {
			searchResults[i].IsFork = *repository.Fork
		}
	}

	return searchResults, nil
}
