/*
Package main implements the twitch command, a shell program similar to git.

Executing it such as `twitch my-command` will attempt to execute the binary
`twitch-my-command`, passing along any provided flags.

Executing it with the help option (-h) displays the full list of available
subcommands, with a one-line description of each.
*/
package main

import (
	"bytes"
	"fmt"
	"log"
	"os"
	"os/exec"
	"strings"

	"github.com/spf13/cobra"

	"code.justin.tv/twitch/cli/pkg/help"
	"code.justin.tv/twitch/cli/pkg/process"
)

var (
	helppp bool
)

func init() {
	rootCmd.Flags().BoolVarP(&helppp, "helppp", "H", false, "extended help (shows subcommands, recursively)")

	// find executables to install as subcommands
	executables, err := findTwitchExecutables()
	if err != nil {
		log.Fatal(err)
	}

	// install executables as subcommands
	for _, e := range executables {
		rootCmd.AddCommand(makeBinarySubcommand(e))
	}
}

var rootCmd = &cobra.Command{
	Use:              "twitch",
	Short:            "Twitch Engineering CLI. Get Shit Done.™",
	TraverseChildren: true,
	Run: func(cmd *cobra.Command, args []string) {
		runRoot(cmd, args...)
	},
}

func runRoot(cmd *cobra.Command, args ...string) {
	if helppp {
		err := runHelppp(cmd, args...)
		if err != nil {
			log.Fatal(err)
		}
		return
	}

	// default to help
	err := cmd.Help()
	if err != nil {
		log.Fatal(err)
	}
}

func runHelppp(cmd *cobra.Command, args ...string) error {
	return help.Helppp(cmd, false, "twitch")
}

func makeBinarySubcommand(name string) *cobra.Command {
	sub := fmt.Sprintf("twitch-%v", name)

	subCmd := &cobra.Command{
		Use:   name,
		Short: getSubCommandDescription(name),
		Run: func(cmd *cobra.Command, args []string) {
			err := process.Exec(sub, args...)
			if err != nil {
				log.Fatal(err)
			}
		},
	}
	subCmd.DisableFlagParsing = true

	// Make help show help of subcommands
	subCmd.SetHelpFunc(func(cmd *cobra.Command, args []string) {
		err := process.Exec(sub, "-h")
		if err != nil {
			log.Fatal(err)
		}
	})

	return subCmd
}

// findTwitchExecutables returns all executable commands available via $PATH which begin with "twitch-"
func findTwitchExecutables() ([]string, error) {
	var out bytes.Buffer
	cmd := exec.Command("bash", "-c", "compgen -A command twitch-")
	cmd.Stdout = &out
	cmd.Stderr = os.Stderr
	err := cmd.Run()
	if err != nil {
		return []string{}, fmt.Errorf("error finding twitch executables: %v", err)
	}

	var cmds []string
	for _, c := range strings.Split(strings.TrimSpace(out.String()), "\n") {
		cmds = append(cmds, strings.TrimPrefix(c, "twitch-"))
	}

	return cmds, nil
}

// getSubCommandDescription parses the first line of `-h` from a given subcommand
func getSubCommandDescription(command string) string {
	var out bytes.Buffer
	cmd := exec.Command(fmt.Sprintf("twitch-%v", command), "-h")
	cmd.Stdout = &out
	err := cmd.Run()
	if err != nil {
		// we expect the command to exit with status code 2,
		// but should not treat this as an error since it is normal for -h
	}

	return help.ExtractDescription(out.String())
}
