package main

import (
	"flag"
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"strings"
	"testing"
	"unicode"

	"github.com/stretchr/testify/assert"
)

// In the following tests, there's a lot going on.
// Please have a look at the following article for a longer explanation:
// http://lucapette.me/writing-integration-tests-for-a-go-cli-application

var update = flag.Bool("update", false, "update golden files")

const binaryName = "twitch"

var binaryPath string

func trimTrailingWhitespace(s string) string {
	lines := strings.Split(s, "\n")
	for i := range lines {
		lines[i] = strings.TrimRightFunc(lines[i], unicode.IsSpace)
	}

	return strings.Join(lines, "\n")
}

func TestMain(m *testing.M) {
	err := os.Chdir("..")
	if err != nil {
		fmt.Printf("could not change dir: %v", err)
		os.Exit(1)
	}

	abs, err := filepath.Abs(binaryName)
	if err != nil {
		fmt.Printf("could not get abs path for %s: %v", binaryName, err)
		os.Exit(1)
	}

	binaryPath = abs

	if err := exec.Command("make", "build").Run(); err != nil {
		fmt.Printf("could not make binary for %s: %v", binaryName, err)
		os.Exit(1)
	}
	os.Exit(m.Run())
}

func TestCLI(t *testing.T) {
	tests := []struct {
		name    string
		args    []string
		golden  string
		wantErr bool
	}{
		{
			"no arguments",
			[]string{},
			"help.golden",
			false,
		},
		{
			"help flag",
			[]string{"-h"},
			"help.golden",
			false,
		},
		{
			"help command",
			[]string{"help"},
			"help.golden",
			false,
		},
		{
			"foo command",
			[]string{"foo"},
			"foo.golden",
			false,
		},
		{
			"foo help flag",
			[]string{"foo", "-h"},
			"foo.help.golden",
			false,
		},
		{
			"foo flag -i",
			[]string{"foo", "-i"},
			"foo.flag_i.golden",
			false,
		},
		{
			"foo flag -x (unknown flag on subcommand)",
			[]string{"foo", "-x"},
			"foo.flag_x.golden",
			true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			cmd := exec.Command(binaryPath, tt.args...)
			fmt.Printf("%+v", cmd)

			cmd.Env = []string{"PATH=.:integration/fixtures/bin:/bin"}
			output, err := cmd.CombinedOutput()
			if !tt.wantErr {
				assert.NoError(t, err)
			}
			actual := string(output)

			golden := NewGoldenFile(t, tt.golden)

			if *update {
				golden.Write(actual)
			}
			expected := golden.Load()

			assert.Equal(t, trimTrailingWhitespace(expected), trimTrailingWhitespace(actual))
		})
	}
}
