package main

import (
	"io/ioutil"
	"os"
	"path/filepath"
	"runtime"
	"testing"

	"github.com/kr/pretty"
)

// Diff compares results
func Diff(expected, actual interface{}) []string {
	return pretty.Diff(expected, actual)
}

// TestFile represents a fixture or golden file for use in integration tests
type TestFile struct {
	t    *testing.T
	name string
	dir  string
}

// NewFixture represents a fixture file
func NewFixture(t *testing.T, name string) *TestFile {
	return &TestFile{t: t, name: name, dir: "fixtures"}
}

// NewGoldenFile represents a golden file
func NewGoldenFile(t *testing.T, name string) *TestFile {
	return &TestFile{t: t, name: name, dir: "golden"}
}

func (tf *TestFile) path() string {
	tf.t.Helper()
	_, filename, _, ok := runtime.Caller(0)
	if !ok {
		tf.t.Fatal("problems recovering caller information")
	}

	return filepath.Join(filepath.Dir(filename), tf.dir, tf.name)
}

// Write writes the test file
func (tf *TestFile) Write(content string) {
	tf.t.Helper()
	err := ioutil.WriteFile(tf.path(), []byte(content), 0644)
	if err != nil {
		tf.t.Fatalf("could not write %s: %v", tf.name, err)
	}
}

func (tf *TestFile) asFile() *os.File {
	tf.t.Helper()
	file, err := os.Open(tf.path())
	if err != nil {
		tf.t.Fatalf("could not open %s: %v", tf.name, err)
	}
	return file
}

// Load loads the test file
func (tf *TestFile) Load() string {
	tf.t.Helper()

	content, err := ioutil.ReadFile(tf.path())
	if err != nil {
		tf.t.Fatalf("could not read file %s: %v", tf.name, err)
	}

	return string(content)
}
