/*
Package config hosts logic to read and parse files in ~/.twitch/ to enable
Go-based twitch subcommands to use that configuration in a standard way.
*/
package config

import (
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"path/filepath"

	"gopkg.in/yaml.v2"

	homedir "github.com/mitchellh/go-homedir"
)

const (
	confDirPath  = ".twitch"
	confFileName = "config.yml"
)

// Config represents common config items
type Config struct {
	Github *githubConfig
}

type githubConfig struct {
	Username *string
	Token    *string
}

// New creates a new config
func New() *Config {
	return &Config{
		Github: &githubConfig{},
	}
}

// Read reads a config from the filesystem
func Read() (*Config, error) {
	e, err := Exists()
	if err != nil {
		return nil, err
	}

	if !e {
		return nil, fmt.Errorf("~/%v not found! Run 'twitch cli configure' to create it.\n", path.Join(confDirPath, confFileName))
	}

	config := &Config{}

	p, err := resolvedPath()
	if err != nil {
		return nil, err
	}

	b, err := ioutil.ReadFile(p)
	if err != nil {
		return nil, err
	}

	err = yaml.Unmarshal(b, config)
	if err != nil {
		return nil, err
	}

	return config, nil
}

// Exists checks if a config file is present
func Exists() (bool, error) {
	p, err := prepare()
	if err != nil {
		return false, err
	}

	_, err = os.Stat(p)
	if err == nil {
		return true, nil
	}

	if os.IsNotExist(err) {
		return false, nil
	}

	return true, err
}

func resolvedPath() (string, error) {
	home, err := homedir.Dir()
	if err != nil {
		return "", fmt.Errorf("Could not find home directory: %v", err)
	}

	return path.Join(home, confDirPath, confFileName), nil
}

func prepare() (string, error) {
	p, err := resolvedPath()
	if err != nil {
		return "", err
	}

	dir := filepath.Dir(p)

	_, err = os.Stat(dir)
	if err == nil {
		return p, nil
	}

	if os.IsNotExist(err) {
		err = os.Mkdir(dir, os.ModeDir|0700)
		if err != nil {
			return "", err
		}
	}

	return p, nil
}

// Write writes a config to the filesystems
func Write(c *Config) error {
	p, err := prepare()
	if err != nil {
		return err
	}

	e, err := Exists()
	if err != nil {
		return err
	}

	b, err := yaml.Marshal(c)
	if err != nil {
		return fmt.Errorf("Could not render config file: %v", err)
	}

	// if the file exists, print out what the user should do (how do we merge arbirary yaml in go?)
	if e {
		fmt.Println()
		fmt.Printf(`It appears that you already have a ~/.twitch/config.yrml file.
We don't want to mess it up, so please add the following key manually:

`)
		fmt.Printf("%s", b)
	} else {

		f, err := os.Create(p)
		defer func() {
			err := f.Close()
			if err != nil {
				fmt.Fprintf(os.Stderr, "Could not close config file: %v", err)
			}
		}()
		if err != nil {
			return fmt.Errorf("Could not create config file: %v", err)
		}

		_, err = f.Write(b)
		if err != nil {
			return fmt.Errorf("Could not write config file: %v", err)
		}
	}

	return nil
}
