/*
Package ghutil provides utility functions to make working with the Github API easier.  Included
 are methods for retrieving oauth2 tokens from credentials, methods for retrieving client objects
 from oauth2 tokens, and methods for checking in robo-changes to github repositories.

 Example of use:

 ```golang

 import (
	 "code.justin.tv/twitch/cli/ghutil"
 )

 username := "user"
 password := "password"
 otp2fa   := "000000"
 authRequest := &ghutil.GithubAuthorizationRequest {
	 Scopes:      []string{"repos"},
	 Note:        "appname",
	 NoteUrl:     "https://git-aws.internal.justin.tv/yourapp",
	 ClientID:    "0123456789abcdef",
	 ClientSecret:"0123456789abcdef",
	 Fingerprint: "",
 })

 //Auth to get an oauth2 token
 token, err := ghutil.GetCLIToken(user, password, "", authRequest)
 if err != nil {
	 if !ghutil.IsOTP(err) {
		 return err
	 }

	 token, err = ghutil.GetCLIToken(user, password, otp2fa, authRequest)
	 if err != nil {
		 return err
	 }
 }

 //Use the token to get a client & repository
 client, err := ghutil.ClientFromToken(token)
 if err != nil {
	 return err
 }

 repo, err := ghutil.RepoFromFullName(client, "twitch/repo")
 if err != nil {
	 return err
 }

 changes := make(map[string]*github.TreeEntry)
 changes["somefolder/somefile"] = &github.TreeEntry {
	 Path: github.String("somefolder/somefile"),
	 Mode: github.String("100644"),
	 Type: github.String("blob"),
	 Content: github.String("This is the inside of the file"),
 }

 //See the interface ghutil.FileTransform for information on making your own transforms

 commitMessage := "Add `somefile`"
 commit, err := ghutil.CreateCommitFromChanges(client, repo, "master", changes, commitMessage)
 if commit == nil || err != nil {
	 //If commit == nil that means there weren't any changes to commit
	 return err
 }

 //Now create a new branch set to the commit, and create a PR from the branch.  This method will return
 //an error if the branch already exists to avoid mangling existing work.  The commit passed in can exist
 //on the server, but it will be created if it doesn't.
 prTitle := "My New PR"
 prMessage := "I made a change!"
 pr, err := ghutil.CreatePullRequest(client, repo, "some-random-branch", "master", prTitle, prMessage, commit)
 if err != nil {
	 return err
 }

 //Now merge the PR (if you want) and delete your branch (if you want)
 ```
*/
package ghutil

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"path"
	"strings"

	"github.com/google/go-github/github"
	"golang.org/x/oauth2"
)

var (
	githubRoot = "https://git-aws.internal.justin.tv"
	githubURL  = githubRoot + "/api/v3/"
	errOTP     = errors.New("Must specify two-factor authentication OTP code")
)

// GithubAuthorizationRequest contains the values needed to auth against GitHub
type GithubAuthorizationRequest struct {
	Scopes       []string `json:"scopes"`
	Note         string   `json:"note"`
	NoteURL      string   `json:"note_url"`
	ClientID     string   `json:"client_id"`
	ClientSecret string   `json:"client_secret"`
	Fingerprint  string   `json:"fingerprint,omitempty"`
}

type githubAuthorizationResponse struct {
	ID             int                            `json:"id"`
	URL            string                         `json:"url"`
	Scopes         []string                       `json:"scopes"`
	Token          string                         `json:"token"`
	TokenLastEight string                         `json:"token_last_eight"`
	HashedToken    string                         `json:"hashed_token"`
	App            githubAuthorizationResponseApp `json:"app"`
	Note           string                         `json:"note"`
	NoteURL        string                         `json:"note_url"`
	UpdatedAt      string                         `json:"updated_at"`
	CreatedAt      string                         `json:"created_at"`
	Fingerprint    string                         `json:"fingerprint"`
}

type githubAuthorizationResponseApp struct {
	URL      string `json:"url"`
	Name     string `json:"name"`
	ClientID string `json:"client_id"`
}

// ClientFromToken builds Client with the given token
func ClientFromToken(token string) (*github.Client, error) {
	token = strings.TrimSpace(token)

	ts := oauth2.StaticTokenSource(
		&oauth2.Token{AccessToken: token},
	)
	tc := oauth2.NewClient(oauth2.NoContext, ts)
	client := github.NewClient(tc)

	base, err := url.Parse(githubURL)
	if err != nil {
		return nil, err
	}

	client.BaseURL = base
	return client, nil
}

// RepoFromFullName returns a repository given a client and fullname
func RepoFromFullName(client *github.Client, fullname string) (*github.Repository, error) {
	var org, n string

	parts := strings.Split(fullname, "/")
	if len(parts) > 1 {
		org = parts[0]
		n = parts[1]
	} else {
		org = ""
		n = parts[0]
	}

	repo, _, err := client.Repositories.Get(org, n)

	return repo, err
}

// GetCLIToken returns a token for the given username and password
func GetCLIToken(username, password, otp string, authRequest *GithubAuthorizationRequest) (string, error) {
	c := http.DefaultClient

	b, err := json.Marshal(authRequest)
	if err != nil {
		return "", fmt.Errorf("Could not create authorization object: %v", err)
	}

	u, err := url.Parse(githubURL)
	if err != nil {
		return "", err
	}

	u.Path = path.Join(u.Path, "authorizations")
	authorizationURL := u.String()
	body := bytes.NewBuffer(b)

	req, err := http.NewRequest("POST", authorizationURL, body)
	if err != nil {
		return "", fmt.Errorf("Could not create authorization request: %v", err)
	}

	req.SetBasicAuth(username, password)

	if otp != "" {
		req.Header.Set("X-Github-Otp", otp)
	}

	resp, err := c.Do(req)
	if err != nil {
		return "", fmt.Errorf("Could not request authorization: %v", err)
	}

	buf := bytes.NewBuffer(nil)
	_, err = buf.ReadFrom(resp.Body)
	if err != nil {
		return "", fmt.Errorf("Could not read response: %v", err)
	}

	if resp.StatusCode != http.StatusCreated {
		if resp.StatusCode == http.StatusUnauthorized {
			otp, ok := resp.Header["X-Github-Otp"]
			if ok && strings.Contains(otp[0], "required") {
				return "", errOTP
			}
		}
		return "", fmt.Errorf("Could not authorize: %v", buf.String())
	}

	r := &githubAuthorizationResponse{}
	err = json.Unmarshal(buf.Bytes(), r)
	if err != nil {
		return "", fmt.Errorf("Could not parse response: %v", err)
	}

	return r.Token, nil
}

// IsOTP returns true if the error is an OTP error
func IsOTP(err error) bool {
	return err == errOTP
}
