package help

import (
	"fmt"
	"os"
	"regexp"
	"strings"
	"text/tabwriter"
	"unicode"

	"golang.org/x/sync/errgroup"

	"github.com/spf13/cobra"

	"code.justin.tv/twitch/cli/pkg/process"
)

type helpLine struct {
	cmd    string
	txt    string
	indent int
}

func Helppp(cmd *cobra.Command, includeFlags bool, cliArgs ...string) error {
	var err error

	// normal help first
	err = cmd.Help()
	if err != nil {
		return err
	}

	fmt.Println("\n\nAll Subcommands:")
	fmt.Println()

	helpLines := recurseHelp(cliArgs, 0, includeFlags)

	w := new(tabwriter.Writer)
	w.Init(os.Stdout, 1, 1, 2, ' ', 0)
	for _, hl := range helpLines {
		line := strings.Repeat("   ", hl.indent)
		line += hl.cmd + "\t"
		line += strings.TrimLeftFunc(hl.txt, unicode.IsSpace)
		fmt.Fprintln(w, line)
	}
	w.Flush()
	fmt.Println()

	return nil
}

func recurseHelp(cmds []string, indent int, includeFlags bool) []helpLine {
	var g errgroup.Group
	helpLines := []helpLine{}

	// capture the help output
	helpCmd := append(cmds, "-h")
	captured, _ := process.Capture(helpCmd[0], helpCmd[1:]...)

	if includeFlags {
		flags := ExtractFlags(string(captured))
		for _, line := range strings.Split(flags, "\n") {
			if line == "" {
				break
			}

			// find flag name
			split := strings.Split(line, " ")
			f, txt := split[0], split[1:]

			for strings.HasSuffix(f, ",") {
				fst := txt[0]
				txt = txt[1:]
				f += " " + fst
			}

			// output the flag in question
			helpLines = append(helpLines, helpLine{
				cmd:    f,
				txt:    strings.Join(txt, " "),
				indent: indent,
			})
		}
	}

	subcommands := ExtractSubcommands(string(captured))
	helpFragments := make([][]helpLine, len(subcommands))

	for i, line := range strings.Split(subcommands, "\n") {
		i, line := i, line // https://golang.org/doc/faq#closures_and_goroutines

		if line == "" {
			break
		}

		// find subcommand name
		split := strings.Split(line, " ")
		sc, txt := split[0], split[1:]

		for strings.HasSuffix(sc, ",") {
			fst := txt[0]
			txt = txt[1:]
			sc += " " + fst
		}

		// output the subcommand in question
		helpFragments[i] = []helpLine{
			helpLine{
				cmd:    sc,
				txt:    strings.Join(txt, " "),
				indent: indent,
			},
		}

		// recurse with deeper indent
		hc := append(cmds, sc)

		g.Go(func() error {
			helpFragments[i] = append(helpFragments[i], recurseHelp(hc, indent+1, includeFlags)...)
			return nil
		})
	}

	if err := g.Wait(); err != nil {
		return nil
	}

	for _, hf := range helpFragments {
		helpLines = append(helpLines, hf...)
	}

	return helpLines
}

func ExtractDescription(usage string) string {
	lines := strings.Split(usage, "\n")
	desc := lines[0]

	// if first line is a Name: or Usage: (urfave/cli) drop until we find name, return next line split
	if regexp.MustCompile(`(?i)(usage|name):\w*`).MatchString(desc) {
		desc = ""
		_, lines = dropUntil(lines, regexp.MustCompile(`(?i)name:\w*`))

		if len(lines) > 0 {
			maybe := lines[0]
			if strings.Contains(maybe, " - ") {
				return strings.Split(maybe, " - ")[1]
			}
		}
	}

	return desc
}

func ExtractSubcommands(usage string) string {
	subcommands := []string{}
	lines := strings.Split(usage, "\n")

	_, lines = dropUntil(lines, regexp.MustCompile(`(?i)(commands|available commands):\w*`))
	subcommands, lines = dropUntil(lines, regexp.MustCompile(`\A\w*\z`))

	return trimAllWhitespace(strings.Join(subcommands, "\n"))
}

func ExtractFlags(usage string) string {
	flags := []string{}
	lines := strings.Split(usage, "\n")

	_, lines = dropUntil(lines, regexp.MustCompile(`(?i)(flags|global options):\w*`))
	flags, lines = dropUntil(lines, regexp.MustCompile(`\A\w*\z`))

	return trimAllWhitespace(strings.Join(flags, "\n"))
}

func dropUntil(lines []string, r *regexp.Regexp) ([]string, []string) {
	var line string
	var dropped []string

	for len(lines) > 0 {
		// pop and shift
		line, lines = lines[0], lines[1:]
		dropped = append(dropped, line)

		if r.MatchString(line) {
			break
		}
	}

	return dropped, lines
}

func trimAllWhitespace(s string) string {
	lines := strings.Split(s, "\n")
	for i := range lines {
		lines[i] = strings.TrimFunc(lines[i], unicode.IsSpace)
	}

	return strings.Join(lines, "\n")
}
