# Latency Instrumentation

Tracking latency metrics allows us to gain a better understanding of the
user-perceived performance of our web applications as measured from the
browser itself. Our current architecture supports multiple levels of granularity:

* [Page Latency](#page-load-instrumentation)
* [Feature Latency](#feature-load-instrumentation)
* [Network Latency](#network-request-instrumentation)
* [Site Performance Dashboard](https://grafana.internal.justin.tv/dashboard/db/site-performance)
* [Site Performance Report](https://modeanalytics.com/twitch/reports/7d3bc9c6d2f2)

## Page Load Instrumentation

### What does it mean for a page to be ready for interaction?

"Ready for interaction" means that the user can meaningfully interact
with a subset of components on the page that are critical to user
interaction. For example, a directory page is ready when it's rendered
the following column and stream preview thumbnails. Refer to the [Page
Load Time Definitions][page_load_time_definitions] for a list of
pages and their criteria.

### How do I report the page load time for a route in Ember?

To opt-in to page load time instrumentation for a route,
implement the `isInteractive()` method that returns `true` if
and only if all its critical components have been loaded:

```js
export default Route.extend({
  // `isInteractive()` should check `didReportInteractive` and
  // return true if the route's conditions have been met.
  isInteractive(didReportInteractive) {
    // TODO
  }
});
```

If you refresh the page and wait a couple seconds, you'll see a red banner
warning you that `isInteractive()` never returns `true`:

![annoying red "unimplemented" banner](images/page-load-time-instrumentation-red-banner.png)

`isInteractive()` receives a single argument called `didReportInteractive`, 
a function that accepts a component name and returns `true` if it's reported
interactive.

For example, here's how you might implement `isInteractive()`
for a directory page:

```js
isInteractive(didReportInteractive) {
  // The {{following-column}} must have loaded!
  return didReportInteractive('following-column') && (
    // There are two possible states for the directory list of thumbnails:
    
    // 1. There are channels live, and we load {{stream-preview}}s.
    didReportInteractive('stream-preview') ||
    
    // 2. There are no channels live, so we render text that says "No channels are live".
    didReportInteractive('beacon:no-channels-directory-channels-are-live')
  );
}
```

If you reload your route and wait a few seconds, the red banner
should be gone. Your route is now reporting its page load time!

### How can I wait for multiple instances of a component?

Let's say you're instrumenting the front page and you need to
wait for 6 "Top Game" components above the fold before the
front page is considered interactive. You can specify a number
of components using a count parameter:

```js
// app/routes/index.js (Front Page)
isInteractive(didReportInteractive) {
  return didReportInteractive('top-game', { count: 6 });
}
```

## Feature Load Instrumentation

Feature-level metrics allow us to identify bottlenecks on a page
and optimize individual components.

### How can I instrument components to report their latency in Ember?

Add the `interactivity-subscription` mixin to your component and define
the `isInteractive` method.

```js
import InteractivitySubscription from 'web-client/mixins/interactivity-subscription';
export default Component.extend(InteractivitySubscription, {
  // `isInteractive()` should check `didReportInteractive` and
  // return true if the route's conditions have been met.
  isInteractive(didReportInteractive) {
    // TODO
  }
});
```

As with route instrumentation, `isInteractive` receives a single
argument called `didReportInteractive`, a function that accepts a
component name and returns `true` if it's reported interactive.

### How can I make my components report that they're interactive?

The `interactivity-subscription` mixin will automatically send a benchmark
to spade when the `isInteractive` returns true, as well as calling
`reportInteractive` to tell the current route or its parent component
that it's ready for user interaction.

If your component does not need to report its individual latency to spade,
but you still depend on it for a parent route/component, then call
`reportInteractive` manually instead of defining more complicated
interactivity conditions through `isInteractive`.

```js
import InteractivityReporting from 'web-client/mixins/interactivity-reporting';
export default Component.extend(InteractivityReporting, {
  // invoke `this.reportInteractive()` when it's ready.
});
```

#### A simple component

In the simplest use-case, your component wraps already-fetched,
passed-in data. Once the element's been added to the template
and rendered, it's ready to accept user interaction:

```js
// app/components/stream-preview.js
didInsertElement() {
  this._super(...arguments);
  run.schedule('afterRender', this, this.reportInteractive);
}
```

#### A component that makes a request

Often, a component will need to fetch data before it's ready
for user interaction. In this case, you must defer the
`this.reportInterative()` call:

```js
// app/components/friend-list.js
export default Component.extend({
  init() {
    this.get('friends').fetchFriendList().then(friends => {
      this.set('friends', friends);
      // NOTE: We still need to wait for Ember to render the
      // fetched data:
      run.schedule('afterRender', this, this.reportInteractive);
    });
  })
}
```

#### Putting it all together

Once you've instrumented all the critical components, your route or component
is ready to report its page latency!

##### Route
```js
// routes/friend-list.js
export default Route.extend({
  isInteractive(didReportInteractive) {
    return didReportInteractive('following-column') &&
      didReportInteractive('friend-list') &&
      didReportInteractive('stream-preview');
  }
});
```

##### Component
```js
// components/friend-list.js
import InteractivitySubscription from 'web-client/mixins/interactivity-subscription';
export default Component.extend(InteractivitySubscription, {
  isInteractive(didReportInteractive) {
    return didReportInteractive('following-column') &&
      didReportInteractive('friend-list') &&
      didReportInteractive('stream-preview');
  }
});
```

#### Verifying that it worked

You can [install the spade-inspector Chrome extension][spade_inspector] to
verify you're sending events as expected.

![Use the spade inspector to confirm you've done it right](images/spade-inspector-demo.gif)

## Network Request Instrumentation

If you want to track the network timings for your API or CDN requests, there is nothing you need to do! All of those requests are automatically tracked for you!

This is done via [network-metrics-js](https://git.xarth.tv/web/network-metrics-js), a library we wrote in Typescript to track and log network data to Spade. This library leverages the User Timing API to get network timings for _all_ requests made to our API and CDN.

## Definitions

A page's **load time** is the time it takes for it to be ready for user
interaction. There are two main types of page loads, App Launch & In-App Navigation:

#### App Launch

This refers to users accessing our site from any source outside the
application, such as the address bar, a bookmark, or an external link.

App Launches influence Twitch's [Global Page Latency][page_latency] metric.

#### In-App Navigation

This refers to users navigating to new pages after they've already loaded
a Single-Page Application (SPA), such as clicking on a channel from a directory page.

In-App Navigation is tracked in our latency data, but it does not influence
the [Global Page Latency][page_latency] metric.

## B-but what about—

No problem! Please direct questions to the #site-performance Slack
channel or site-performance@twitch.tv

[page_latency]: https://docs.google.com/document/d/1nuKKycVhpiNEK9uHJJ9UC2gHK1h5VZQ6wd5rIZM6Ht4/edit
[page_load_time_definitions]: https://docs.google.com/document/d/1RQZv9BRT9i2Q-rqYLd3x36CG4yymBbkPcnCRBnjFWOI/edit
[spade_inspector]: https://chrome.google.com/webstore/detail/spade-inspector/mhjpfcncekeedcbagjlkchokmggimbbl
