set -e

# Usage: bash script.sh <environment> <branch>
#
#   Example: bash script.sh clean-staging-ffa-999 super-cool-feature

# Setup
#   Github token
#
#     1. Create a personal access token on https://git-aws.internal.justin.tv/settings/tokens
#     2. Personal Settings > Personal access tokens > Generate new token
#     3. Under `Select scopes`, only check `repo_deployment`
#     4. Set GITHUB_DEPLOYMENT_ACCESS_TOKEN in your shell
#
#   jq
#
#     1.  `brew install jq`

ENVIRONMENT="$1"
BRANCH="$2"

if [[ -z "$GITHUB_DEPLOYMENT_ACCESS_TOKEN" ]]; then
  echo "Missing GITHUB_DEPLOYMENT_ACCESS_TOKEN" >&2
  exit 1
fi

if [[ -z "$ENVIRONMENT" ]]; then
  echo "Missing ENVIRONMENT" >&2
  exit 1
fi

if [[ -z "$BRANCH" ]]; then
  echo "Missing BRANCH" >&2
  exit 1
fi

if echo "$ENVIRONMENT" | grep -q production; then
  echo "ERROR Do not use this to deploy to production environments!" >&2
  exit 1
fi

#
# Wait for the build to finish on the branch
#

branch_sha="$(curl -s -H "Authorization: token $GITHUB_DEPLOYMENT_ACCESS_TOKEN" https://git-aws.internal.justin.tv/api/v3/repos/web/web-client/git/refs/heads/$BRANCH \
  | jq -r '.object.sha')"

commit_message="$(curl -s -H "Authorization: token $GITHUB_DEPLOYMENT_ACCESS_TOKEN" https://git-aws.internal.justin.tv/api/v3/repos/web/web-client/git/commits/$branch_sha | \
  jq -r '.message')"

if [[ -z "$branch_sha" ]]; then
  echo "ERROR: Github does not know about branch $BRANCH" >&2
  exit 1
fi

echo "Git branch: $BRANCH sha: $branch_sha"
echo "Git commit message: $commit_message"

while true; do
  build_status="$(curl -s -H "GithubAccessToken: $GITHUB_DEPLOYMENT_ACCESS_TOKEN" "https://clean-deploy.internal.justin.tv/v1/build?owner=web&name=web-client&sha=$branch_sha")"
  if [[ $build_status == "null" ]]; then
    echo "`date` Build is queued..."
    sleep 5
  else
    percent="$(echo "$build_status" | jq -r '.percent')"
    if [[ $percent -lt 100 ]]; then
      estimated_duration="$(echo "$build_status" | jq -r '.estimated_duration')"
      duration="$(echo "$build_status" | jq -r '.duration')"
      delta=$(( ($estimated_duration - $duration) / 1000))
      minutes=$(($delta / 60))
      seconds=$(($delta % 60))

      echo "`date` Building... Percent: $percent% ETA: ${minutes}m${seconds}s"
      sleep 5
    else
      echo "$branch_sha finished"
      break
    fi
  fi
done


#
# Sanity check build status
#

status="$(curl -s -H "Authorization: token $GITHUB_DEPLOYMENT_ACCESS_TOKEN" https://git-aws.internal.justin.tv/api/v3/repos/web/web-client/commits/$branch_sha/status)"
state="$(echo "$status" | jq -r '.state')"

if [[ "$state" == "success" ]]; then
  echo "State is $state. Continuing with deployment..."
elif [[ "$state" == "failure" ]]; then
  target_url="$(echo $status | jq -r '.statuses[0].target_url')"
  echo "ERROR Build failed. Check ${target_url}console" >&2
  exit 1
else
  echo "ERROR Unhandled state $state" >&2
  exit 1
fi

#
# Create deployment and wait for it to finish
#

deployment="$(curl -s 'https://clean-deploy.internal.justin.tv/v1/deployments' \
  -H "GithubAccessToken: $GITHUB_DEPLOYMENT_ACCESS_TOKEN" \
  --data "owner=web&repo=web-client&ref=$branch_sha&branch=$BRANCH&environment=$ENVIRONMENT&description=$DESCRIPTION&codeReviewURL=&severity=&triggerSmoca=")"

deployment_id="$(echo $deployment | jq -r '.id')"

while true; do
  deployment_status="$(curl -s -H "GithubAccessToken: $GITHUB_DEPLOYMENT_ACCESS_TOKEN" "https://clean-deploy.internal.justin.tv/v1/deployments/$deployment_id/status")"
  state="$(echo $deployment_status | jq -r '.state')"
  if [[ $state == "error" ]]; then
    echo "ERROR: Deployment $deployment_id failed with state $state" >&2
    exit 1
  elif [[ $state == "unknown" ]]; then
    echo "`date` Deployment $deployment_id is queued"
    sleep 5
  elif [[ $state == "pending" ]]; then
    echo "`date` Deployment $deployment_id is pending"
    sleep 5
  elif [[ $state == "success" ]]; then
    echo "`date` Deployment $deployment_id succeeded!"
    exit 0
  else
    echo "ERROR: Unhandled deployment state $state" >&2
    echo "Response body: $deployment_status" >&2
    exit 1
  fi
done
