Terraform as Baremetal
======================

Please see the [general terraform documentation](https://git.xarth.tv/twitch/docs/blob/master/aws/terraform.md) first.

This guide is about how to use terraform with Puppet.

Puppet in AWS **requires** two pieces of information to be set:

1. The FQDN of the box must be set to `<something>.us-west2.justin.tv`.
2. There must be a `cluster` fact set.

This is generally accomplished by using the following userdata when creating
an instance.

```
resource "aws_instance" "example" {
  ...
  user_data = <<END_OF_STRING
#cloud-config
hostname: "godoc-${count.index}"
fqdn: "godoc-${count.index}.prod.us-west2.justin.tv"
manage_etc_hosts: true
runcmd:
 - sleep 1
 - echo "cluster=<cluster_name>" > /etc/facter/facts.d/cluster.txt
 - echo "twitch_environment=production" > /etc/facter/facts.d/twitch_environment.txt
 - echo "clean=true" > /etc/facter/facts.d/clean.txt
 - puppet agent --test
END_OF_STRING
}
```

In addition it is highly recommended that you set the following facts:

1. `clean=true` to indicate that the server is "urgent-pizza clean".
2. `twitch_environment=<environment>` to indicate puppet differences by
   environment.
   
Additional facts can be used as desired.

Puppet seeing the `cluster` fact will load the matching file in the
`hiera/cluster/` folder and process the settings set there.

For an example see:

* [Godoc Terraform Settings](https://git.xarth.tv/release/terraform/blob/master/godoc/godoc.tf)
* [Godoc Puppet Declaration](https://git.xarth.tv/systems/puppet/blob/master/hiera/cluster/godoc.yaml)

## Dynamic Environments

If you wish to create dynamic environments there is a bit of extra work
required.

* You must register the environment in consul:
   ```
   resource "consul_keys" "web-env" {
       datacenter = "sfo01"
       key {
           name = "env-registration"
           path = "dynamic-envs/<org>/<owner>/clean-${var.env.name}"
           value = "{}"
           delete = true
       }
   }
   ```

* You must install and call update-deployed-version to set the initial
   deployed version.
   
   ```
   resource "consul_keys" "web-known-version" {
       datacenter = "sfo01"
       key {
           name = "known-version"
           path = "deployed-version/web/web/clean-production"
       }
   }
   
   
   resource "aws_instance" "app" {
     ...
     provisioner "local-exec" {
         command="CONSUL_HOST='consul.internal.justin.tv' update-deployed-version -env clean-${var.env.name} -repo web/workers -commit ${consul_keys.web-workers-known-version.var.known-version}"
     }
   }
   ```
