# [Production-ready Go services from scratch](shipping_a_new_app.md)

* [Deploying "Hello World" with AWS Beanstalk](#deploying-hello-world-with-aws-beanstalk)
	* [Glossary](#glossary)
	* [Preparation](#preparation)
	* [Produce application versions during your build](#produce-application-versions-during-your-build)
		* [Define Docker image](#define-docker-image)
		* [Generate JSON configuration](#generate-json-configuration)
	* [Create an environment](#create-an-environment)
		* [Development/Staging](#developmentstaging)
		* [Production](#production)
		* [Troubleshooting](#troubleshooting)
	* [Deploy to an environment](#deploy-to-an-environment)
	* [Customizing your configuration](#customizing-your-configuration)

## Deploying "Hello World" with AWS Beanstalk

Elastic Beanstalk (EB) is an AWS service which aims to make it easier to deploy common applications (i.e. web apps). By packaging your application in an EB-compatible way, you can gain the following features for your application.

- Provision instances across availability zones
- Rolling updates during deploys
- Load balanced endpoint access for clients
- Automatically scale up/down as load changes
- Single command to spin up new environments
- Basic log access without SSH

You could get these features by using various AWS services directly, but EB orchestrates them for you. So if your service hosts an HTTP server which you'd like to expose to other applications easily as a single endpoint, then you are a good candidate.

### Glossary

- **Application**: A logical collection of EB components, including environments, versions, and environment configurations.
- **Environment**: A specific running instance of an application.
- **Application Version**: A specific, labeled iteration of an application that represents a functionally consistent set of deployable application code. Created as a `.zip` archive.

### Preparation

Before doing anything else, use the [EB web console](https://us-west-2.console.aws.amazon.com/elasticbeanstalk/home?region=us-west-2) to create your application. After supplying a name and optional description, you will be prompted to create the first environment for your app.  **Don't do this**. We will be using command line tools for this step later.

- [ ] Create your EB application, but don't define an environment.

Next install the [EB CLI](http://docs.aws.amazon.com/elasticbeanstalk/latest/dg/eb-cli3-install.html). If you want to install this on your Mac, it should be as simple as

```bash
brew update
brew install awsebcli
eb --version
# EB CLI 3.4.6 (Python 2.7.6)
```

- [ ] Install command line tool

With your application defined, run the command `eb init` inside of your project root.  This will walk you through an interactive set of choices for your configuration.

- When prompted to select an application, choose the one you just created.
- When prompted to select a platform, choose "Multi-container Docker (Generic)"

A `.elasticbeanstalk` folder is then created and these options will be passed in to future calls using the EB command line tool. This directory should be automatically added to the git ignores for your repo.

- [ ] Create base configuration using `eb init`

### Produce application versions during your build

EB usage at Twitch integrates with our existing deployment pipeline, so you will continue to use Jenkins and our deploy UI. The change we need to make is for each Jenkins build to automatically upload an EB-compatible version of your application.

#### Define Docker image

EB apps run via Docker, so you will need to define a Dockerfile capable of running your application.  The details of [Docker syntax](https://docs.docker.com/reference/builder/) are out of scope of this guide, but by using [manta](https://git.xarth.tv/release/manta-style) for your build process your docker image will likely only need your binary and a description of how to execute it.

```docker
# dta/myapp.docker
FROM ubuntu:trusty
ADD .manta/myapp myapp
ENTRYPOINT ["/myapp"]
```

If your app requires making requests to `https://` urls, you need to install certs on your docker container:

```docker
# dta/myapp.docker
FROM ubuntu:trusty
RUN apt-get update && apt-get install ca-certificates -y
... # rest of docker
```

- [ ] Create a Dockerfile for your binary and save it to `dta/myapp.docker`

#### Generate JSON configuration

Each EB application version needs to contain a `Dockerrun.aws.json` file. This file tells EB how to run the container for a particular version. For example, you can instruct it to map a port on the host to a port inside the container to route network traffic.  A simple config for commit `deadbeef` of your repository may look like:

```json
{
	"AWSEBDockerrunVersion": 2,
	"containerDefinitions": [
		{
			"name": "myapp",
			"image": "docker-registry.dev.us-west2.twitch.tv/myapp:deadbeef",
			"essential": true,
			"memory": 256,
			"portMappings": [
				{
					"hostPort": 80,
					"containerPort": 8000
				}
			]
		}
	]
}
```

Ideally the commit SHA should not be hardcoded into your file. The DevTools team is current working to make this simpler, but for now you need to generate this json file as part of your build. Since the `$GIT_COMMIT` environment variable is exposed to scripts run during your Jenkins build job, you can use a script such as the following to create your docker image, push it to our registry, and then package the json config for that image.

```bash
# scripts/beanstalk.sh
set -e -x

IMAGE=docker-registry.dev.us-west2.twitch.tv/myapp:$GIT_COMMIT

docker build -f dta/myapp.docker -t $IMAGE .
docker push $IMAGE

cat <<EOF > Dockerrun.aws.json
{
	...
	"image": "$IMAGE",
	...
}
EOF

zip app.zip Dockerrun.aws.json
```

This script creates an `app.zip` file ready to be uploaded to EB. Be sure to make the script executable with `chmod +x`. In your Jenkins build job, use the `uploadBeanstalkApp` helper function to perform this upload after you execute your shell script.

```groovy
// jenkins.groovy
wrappers {
	credentialsBinding {
		string 'AWS_ACCESS_KEY', 'jenkins-beanstalk-aws-access-key'
		string 'AWS_SECRET_KEY', 'jenkins-beanstalk-aws-secret-key'
	}
}

steps {
	shell 'manta -v'
	shell 'scripts/beanstalk.sh'
	uploadBeanstalkApp 'my-eb-app-name', 'app.zip'
}
```

After your build runs, you can use the EB web console to view the list of available application versions. If your application version isn't there, check the following:
- Make sure your changes are on the `master` branch. The changes to `jenkins.groovy` only get applied if they are on the `master` branch.
- Check the Jenkins logs to see if the `uploadBeanstalkApp` command tried to run. If it didn't even seem to run:
  - Go to the `job_dsl_seed` project in Jenkins and trigger a build. Supposedly this project builds whenever someone changes a jenkins change on master, but there's possibly a caching bug with it not running automatically.
  - Re-run your build and see verify the upload is happening.


- [ ] Trigger a build for your project, confirming the version is listed for your application.
- [ ] Go bug them in #devtools that this process isn't automated yet. :trollface:

### Create an environment

Now that your application has an available version, we can create an environment configured to run that version. The tricky part here is selecting the correct VPC, subnet, and security group settings. At the time of writing, the following examples will correctly provision environments in our Development and Production VPCs in the us-west-2 region.

When terraform gains native EB support, we will likely move to using that instead of running these commands by hand. Until then, each command needs only to be run once for each environment. It will create the environment and run the specified application version.

Use the [EB CLI documentation](http://docs.aws.amazon.com/elasticbeanstalk/latest/dg/eb3-create.html) for details about the available flags.

- [ ] Create an environment using the EB CLI

Check the jenkins logs for the "VersionLabel" of your build. Use that as the `--version` argument in the commands below.

#### Development/Staging

```
eb create myapp-staging \
	--version deadbeef \
	-ip Twitch_Beanstalk_Role \
	-i m3.medium \
	--scale 1 \
	--vpc --vpc.id vpc-1e13b17b \
	--vpc.ec2subnets subnet-d4a423b1,subnet-67ad0810,subnet-894ab1d0 \
	--vpc.elbsubnets subnet-d4a423b1,subnet-67ad0810,subnet-894ab1d0 \
	--vpc.publicip \
	--vpc.securitygroups sg-f1edb194
```

#### Production

```
eb create myapp-production \
	--version deadbeef \
	-ip Twitch_Beanstalk_Role \
	-i m3.xlarge \
	--scale 4 \
	--vpc --vpc.id vpc-0213b167 \
	--vpc.ec2subnets subnet-faa4239f,subnet-88a207ff,subnet-9c4ab1c5 \
	--vpc.elbsubnets subnet-faa4239f,subnet-88a207ff,subnet-9c4ab1c5 \
	--vpc.publicip \
	--vpc.securitygroups sg-27d18d42
```

#### Troubleshooting

At this point you should have a running service, but lots of things could go wrong. Your best bet for debugging is to SSH into the instance EB created for you and look at the Docker logs.

Search for your instance(s) using the EC2 web console and SSH using your specified keypair and the `ec2-user` user.

`ssh -i ~/.ssh/mykeypair ec2-user@1.2.3.4`

Once on the host, use `sudo docker ps -a` to see every container that has run on the box.  If yours failed to start, you should see it listed with an exit code. Take the "container id" and use it to view your logs.

`sudo docker logs -f $ID`

If your application is failing to start due to missing environment variables, you can provide those by either:

- Using the EB web console to define the variables for your enviroment: "Configuration -> Software Configuration".
- Providing the environment variables when creating the EB environment using the `--envars` flag.
- Add the `ENV` command to your `myapp.docker` file.

### Deploy to an environment

Once you have confirmed that creating an environment from scratch results in a healthy deployment, next you need to teach your deploy job how to refresh the environment. In your `deploy.json` file, add your new environments using the exact name of your EB environment.

```json
{
	"environments": {
		"myapp-staging": {},
		"myapp-production": {}
	}
}
```

In your Jenkins deploy job definition, use the `deployBeanstalkApp` helper to accomplish this.

```groovy
// jenkins.groovy
wrappers {
	credentialsBinding {
		string 'AWS_ACCESS_KEY', 'jenkins-beanstalk-aws-access-key'
		string 'AWS_SECRET_KEY', 'jenkins-beanstalk-aws-secret-key'
	}
}

steps {
	deployBeanstalkApp 'myapp'
}
```
If there exists a normal deploy job already, create a new job for beanstalk. Under your environment for `deploy.json`, specify the job name.

```json
{
	"enviroments": {
		"myapp-staging": {
			"job": "new-jenkins-job"  
		},
		"myapp-production": {}
	}
}
```

- [ ] Use the Twitch deploy UI to update the version running on your EB enviroment.

### Customizing your configuration

There are [various configuration options](http://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options.html) you can customize when packaging an application version. For example, by default the load balancer health check simple trys to connect to TCP port 80.  Let's get it to make an HTTP request to our app's debug endpoint.

Create a file at `.ebextensions/myapp.config`. Looking at the above linked options choices, we see that the [health check setting](http://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options.html#command-options-general-elasticbeanstalkapplication) uses the namespace `aws:elasticbeanstalk:application` and the name `Application Healthcheck URL`. To change the setting to hit the `/debug/running` endpoint, edit the file to look like:

```yaml
# .ebextensions/myapp.config
option_settings:
  - namespace: aws:elasticbeanstalk:application
    option_name: Application Healthcheck URL
    value: /debug/running
```

Now append your beanstalk script to include that file in your application version zip.

```bash
# scripts/beanstalk.sh
zip app.zip .ebextensions/myapp.config
```

The next time you deploy, you should be able to use the EB web console to navigate to the "Configuration -> Load Balancing" section of your enviroment and see the path you set.

- [ ] Confirm your configuration was applied after a deploy using the web console.
