# [Production-ready Go services from scratch](shipping_a_new_app.md)

* [Decoupling external systems](#decoupling-external-systems)
* [Testing your API without using real backends](#testing-your-api-without-using-real-backends)

## Decoupling external systems

It is very likely that your application will communicate with an external system such as a database, another Twitch service, or both at the same time.

We aim to decouple our HTTP endpoints from these integrations by implementing requests to them in a separate package. A tenant of Go testing is that tests in our `api` package should only exercise behavior in that package. To enable isolating the endpoint logic, we define our backend as an interface provide an implementation to the server when it starts.

```go
// backend/backend.go
package backend

import (
	"net/http"

	"golang.org/x/net/context"
)

type Backender interface {
	DoThing(context.Context) error
}

type Backend struct {
	Client *http.Client
}

func NewBackend(client *http.Client) *Backend {
	return &Backend{
		Client: client,
	}
}

func (b *Backend) DoThing(ctx context.Context) error {
	// make http requests using b.Client.Get()
	return nil
}

// api/api.go
type Server struct {
	*twitchhttp.Server
	Backend backend.Backender
}

func NewServer(b backend.Backender) (*Server, error) {
	s := &Server{
		Backend: b,
	}

  // etc...
}

// api/thing.go
func (s *Server) doThing(ctx context.Context, c web.C, w http.ResponseWriter, r *http.Request) {
	err := s.Backend.DoThing(ctx)
	if err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	fmt.Fprintf(w, "%s", "OK")
}

// main.go
b := backend.NewBackend(&http.Client{})

server, err := api.NewServer(b)
// etc...
```

In this example, the backend is generically named as `Backend` and wraps an `http.Client` but this is only for example purposes. It should be easy to imagine attaching multiple "backend" interfaces to your `Server` struct, providing an implementation for each during application boot. Each implementation would depend on different client libraries to communicate to the external system.

## Testing your API without using real backends

In the above example, your `/v1/thing` endpoint now dictates its behavior based on the result of a call to your backend.  Since we defined our backend as an interface, we can provide our test with a different implementation.

Here we introduce two libraries to make testing easier, especially if you are comfortable writing Ruby tests with rspec.

- [github.com/stretchr/testify/mock](https://github.com/stretchr/testify) - Enables you to define mock implementations of your backend functions, providing predictable output inside each test.
- [github.com/smartystreets/goconvey](https://github.com/smartystreets/goconvey) - Enables expressing your tests in BDD-style blocks using strings as context descriptors.

```go
// backend/mock_backend.go
package backend

import (
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

type MockBackend struct {
	mock.Mock
}

func (m *MockBackend) DoThing(ctx context.Context) error {
	args := m.Called(ctx)
	return args.Error(0)
}

// api/thing_test.go
package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/myorg/myrepo/backend"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestDoThing(t *testing.T) {
	Convey("when hitting /v1/thing", t, func() {
		backend := &backend.MockBackend{}
		server, err := NewServer(backend)
		So(err, ShouldBeNil)

		testServer := httptest.NewServer(server)
		defer testServer.Close()

		Convey("and the backend succeeds", func() {
			backend.On("DoThing", mock.Anything).Return(nil)

			resp, err := http.Get(fmt.Sprintf("%v/v1/thing", testServer.URL))
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, 200)
		})
		Convey("and the backend fails", func() {
			backend.On("DoThing", mock.Anything).Return(fmt.Errorf("failed"))

			resp, err := http.Get(fmt.Sprintf("%v/v1/thing", testServer.URL))
			So(err, ShouldBeNil)
			So(resp.StatusCode, ShouldEqual, 500)
		})
	})
}

```

In your test, instantiate a mock backend and an API server with the mock as a parameter.  Using the `httptest` package, you can start up an in-memory instance of your endpoints and send requests to them.

- [ ] Implement API tests while mocking the backend implementation
