# [Production-ready Go services from scratch](shipping_a_new_app.md)

* [Deploying "Hello World" with internal tooling](#deploying-hello-world-with-internal-tooling)
	* [Puppet module](#puppet-module)
		* [Hiera cluster file](#hiera-cluster-file)
		* [Pass application environment to your process](#pass-application-environment-to-your-process)
		* [Production node classification](#production-node-classification)
	* [Pre-production environments](#pre-production-environments)
	* [Deploy configuration](#deploy-configuration)
		* [Shipping your first version](#shipping-your-first-version)

## Deploying "Hello World" with internal tooling

### Puppet module

The current Twitch-default is to deploy updated binaries to long-lived servers. These servers can be either bare metal hardware in our various datacenters around the world, or AWS instances. We use the configuration management tool Puppet to define what should be installed on these servers.

Due to this, writing a new Puppet module for your application is required.  This puppet module is be responsible for:

- Passing the application environment into your process.
- Registering the host as a deploy target via consul
- Defining how to run your process, likely via daemontools. Example:

```puppet
# modules/my_app/manifests/install.pp
daemontools::supervise { "my_app":
  daemon_dir     => '/var/lib/service',
  user           => 'nobody',
  syslog         => 'local3',
  enable_cgroups => true,
  wd             => '/opt/twitch/my_app/current',
  daemon         => "./my_app 2>&1",
  predaemon      => [
    "export ENVIRONMENT=${deploy_env}",
    "source /etc/profile.d/proxy.sh"
  ]
}
```

**Note**: Bare metal servers in Twitch datacenters need to utilize an HTTP proxy to talk to external systems. Forgetting to source `proxy.sh` could cause the inability to report errors to services such as Rollbar.

- [ ] Define puppet module for your service [using DevTools documentation](https://git.xarth.tv/twitch/docs/blob/master/release/puppet-consul.md).

#### Hiera cluster file

Puppet modules are attached to machines using hiera cluster. The minimum required configuration looks like:

```yaml
# hiera/cluster/my-app.yaml
---
classes:
  - my_module
```

- [ ] Define hiera cluster for your service

#### Pass application environment to your process

In order to have environment-specific config, your application needs to know what environment it is running in.  A pattern to do this is to have a "puppet fact" override the default production setting.  You will see how to set this fact for staging hosts in a later section.

```puppet
# modules/my_app/manifests/params.pp
class my_app::params {
  $deploy_env = pick($::my_app_env, 'production')
}
```

This snippet shows how to set `$deploy_env` to the value of the named fact, or "production" if the fact is null.

#### Production node classification

If you are planning to host production on bare metal servers, you need to add a new hostname mapping. Skip this step if your service will run on AWS instances exclusively.

Edit [site.pp](https://git.xarth.tv/ops/puppet/blob/master/manifests/site.pp), adding a new mapping to define which hosts should be included in your cluster.

```puppet
node /^my-hostname-.*/ {
  $cluster = 'my-app'
  $twitch_environment = 'production'
  hiera_include('classes')
}
```

- [ ] If your production bare-metal hosts are ready, add your hostname regular expression to site.pp

### Pre-production environments

[Terraform](https://www.terraform.io/docs/index.html) is a tool that aids in defining and provisioning AWS resources. Teams use it to create staging environments.

Add the following commands to your development environment setup logic:

```bash
apt-get install -y unzip
wget https://dl.bintray.com/mitchellh/terraform/terraform_0.5.3_linux_amd64.zip
sudo unzip -d /usr/local/bin terraform_0.5.3_linux_amd64.zip
```

- [ ] Install Terraform in your development environment

You'll need to define provisioning rules for a host configured to apply the puppet module you just created. In the [release/terraform](https://git.xarth.tv/release/terraform) repository there exists a `puppet_instance` convenience wrapper to do exactly this.

```
# terraform/staging.tf
provider "aws" {
  region = "us-west-2"
}

module "puppet_instance" {
  source = "git::git+ssh://git@git.xarth.tv/release/terraform.git//puppet_instance"
  hostname_prefix = "my-app"
  cluster = "my-app"
  owner = "my-team"
  facts = "my_app_env=staging"
}
```

Twitch's deploy service has a "dynamic environment" feature, allowing it to discover available environments via a lookup in Consul.  Instead of statically declaring your new environment, you can hook into this feature by writing your environment key when the VM is provisioned.

```
# terraform/staging.tf
provider "consul" {
  address = "consul.internal.justin.tv"
  datacenter = "sfo01"
  scheme = "http"
}

resource "consul_keys" "dynamic-env" {
  datacenter = "sfo01"
  key {
    name = "env-registration"
    path = "dynamic-envs/my_org/my_repo/staging"
    value = "{}"
    delete = true
  }
}
```

In order to create the instance, run the following commands:

```bash
cd terraform/
export AWS_ACCESS_KEY=foo
export AWS_SECRET_KEY=bar
terraform get # downloads external modules
terraform plan -module-depth 1 # shows you what will happen
terraform apply # creates the VM
```

Make sure to add `.terraform` and `terraform.tfstate.backup` to the project's `.gitignore`.

- [ ] Add terraform files to your repository and provision a staging AWS instance using your new puppet module

When the terraform run completes, you will see output printed to your console that shows you the hostname of your new server. It looks like:

```
name: "" => "myapp-8ac7b07d.dev.us-west2.justin.tv"
```

You should be able to SSH into this server using your LDAP credentials. If you are quick, the puppet run which is applying your module will still be running.

- [ ] SSH into your new server and tail `/var/log/syslog` to confirm the puppet run for your module completes with no errors.
- [ ] Search [Consul](http://consul.internal.justin.tv/ui/dist/#/us-west2/services) for your "deploy-target" service to confirm your host has been registered.

When the puppet run succeeds, your process watchdog will be defined on the host but your binary has not yet been installed.

- [ ] Confirm that your app is trying to run by tailing `/var/log/jtv/myapp.log` and seeing `setuidgid: fatal: unable to run ./myapp: file does not exist`

### Deploy configuration

Now that you have a fresh AWS instance configured via your puppet module, you are ready to deploy your app.

Courier is our standardized deploy tool, and we use Jenkins to execute it.  [Define your deploy job in the same file as your build job](https://git.xarth.tv/twitch/docs/blob/master/release/deploy.md#3-define-deploy-job-in-jenkins).

- [ ] Add a deploy job to your `jenkins.groovy`

Each repo is expected to include a `deploy.json` at its root, configuring the deploy service. Detailed configuration options can be found [here](https://git.xarth.tv/twitch/docs/blob/master/release/deploy.md#5-configure-deploy-details-in-your-repository).

```json
{
  "artifact": "tar",
  "job": "myorg-myrepo-deploy",
  "environments": {
    "production": {}
  },
  "consul_services": [
    "deploy-target-myapp"
  ],
  "restart": {
    "style": "gradual",
    "concurrency": 1,
    "service": "my_app"
  }
}
```

- [ ] Create `deploy.json` in your repository

Do you want messages posted to slack when you deploy? [Of course you do!](https://git.xarth.tv/twitch/docs/blob/master/release/deploy.md#5-configure-deploy-details-in-your-repository)

```json
  },
  "slack": {
    "channel": "#web",
    "branches": {
      "master": {
        "failure": true,
        "pending": false,
        "success": true
      }
   },
   "environments": {
      "*": {
        "failure": true,
        "pending": false,
        "success": true
      }
    }
  }
}
```

- [ ] Add slack to your `deploy.json`

#### Shipping your first version

Using our [deploy UI](https://clean-deploy.internal.justin.tv/#/home) you should be able to navigate to your project using the left menu.

The environments listed in `deploy.json` and any dynamic environments should appear as target choices in the "Deploy" dropdown menu.

- [ ] Deploy your master branch to staging, and confirm that the process starts and stays running.
