# [Production-ready Go services from scratch](shipping_a_new_app.md)

* [Application layout](#application-layout)
* [Loading configuration](#loading-configuration)
* [Vendoring dependencies](#vendoring-dependencies)

## Application layout

Instead of approaching the structure of your Go application using a MVC mindset (ala Rails), think of the components which make up web application built to serve data.

- HTTP endpoints, request parsing, response encoding
- Communication with external systems (such as a database), executing requests/queries, connection pooling.
- Business logic for Twitch products, domain modeling, etc.

Each of these components translates into what Go calls a "package", which is just a subfolder in your repository. Unit tests in Go are meant to exercise the logic in the package they are defined in, so tests in your API package should not depend on database queries or business logic. You'll see patterns for decoupling these components for testing in a later section.

For example, if you are building an app which talks to a single postgres database, you can start with a simple layout such as:

```
myapp/
  api/
    api.go
  backend/
    backend.go
  logic/
    logic.go
  main.go
```

## Loading configuration

If your app has simple configuration needs, consider using the [common/config](https://git.xarth.tv/common/config) library.  It allows your app to easily load configuration from either command line flags, or environment variables.

The following app could be executed using either `go run main.go -my-config foobar` or `MY_CONFIG=foobar go run main.go`.

```go
// main.go
package main

import (
	"log"

	"code.justin.tv/common/config"
)

func init() {
	config.Register(map[string]string{
		"my-config": "my-default",
	})
}

func main() {
	config.Parse()

	log.Print(config.Resolve("my-config"))
}

```

If your app requires complex configuration, then you may need to implement reading a custom config file.

- [ ] Decide on your configuration needs and implement them into `main.go`

## Vendoring dependencies

Now that we have an example of using an external library, we need to vendor it inside our project. You already ran `go get` locally to fetch the libforerunner package, so now we simply use the `godep` tool to save a snapshot of it in our project.

```bash
godep save ./...
```

You will need to run this command each time you use a new external library. For more examples of `godep` workflows, including upgrading packages you have already vendored, refer to the [godep documentation](https://github.com/tools/godep).

- [ ] Vendor your dependencies using `godep` and commit the entire `Godeps` directory it created.
