# Ruby Code Style Guide

## TL;DR

- Don't align things vertically.
- The linter will guide you through the rest.

That's pretty much it. Reading further is optional.

## Approach

We think about the following values when styling our code:

1. ***Consistency***
1. ***Maintainability***
1. ***Readability***

Ruby style is a controversial topic. We choose to enforce a small subset of rules to keep our codebases maintainable across our growing pool of developers.

The linting utility we use is [RuboCop](https://github.com/bbatsov/rubocop). Our current configured version is 0.30.1, and the canonical configuration is located [here](https://git.xarth.tv/web/web/blob/master/.rubocop.yml). Depending on your project, the linter may be integrated in your code review workflow configuration, your build process, or both. New and smaller projects may choose to do things manually.

## Rules

### Maximum Line length:

- Don't exceed 120 characters.

Use this flexibility to choose `good_variable_names`, not as an excuse to write `unreadably.long(*sequences).of(method).ca { |l| l.s }.together`. Just because you can doesn't mean you should.

### Whitespace

- Indent with 2 spaces.
- No hard tabs.
- Do not align things vertically. No consecutive runs of more than 1 space after the indent.
- No trailing whitespace at the end of lines.
- Single line break at the end of the file.
- Space before and after the `#`.

### Assignments

- Space before and after the `=`.
- Do not align the values of assignments.

Imagine yourself as the next person touching this code. Your project requires that you add `yet_another_thing = 333`.

```ruby
# bad
something     =  1
another_thing = 22

# good
something = 11
another_thing = 22
```

### Operators

- Prefer `&&` and `||` over `and` and `or`.
- Space pad binary operators.
- Do not space pad unary operators.

### Blocks

- Use curly braces `{}` if the block exists entirely on one line, use `do`/`end` if it doesn't.
- For single-line blocks pad the contents with one space. Use `{}` for empty blocks.
- Precede single-line blocks with a space.

```ruby
(1..9).inject({}) do |total, num|
  total.merge(
    "row #{num}" => (1..num).map { |x| x * 2 + 1 }
  )
end
```

- String interplation blocks do not need to be space padded.

```ruby
# ok
puts "hello #{user}"
```

### If/Unless

- Never use `else` with `unless`.

### Method Calls

- Never put a space between a method name and the opening parenthesis.
- When chaining methods across lines, put the `.` at the end of the first line.

Using a trailing `.` lets us know that the statement is incomplete. Putting it on the following line makes the code difficult to work with in a REPL/console and makes it possible to instert a debugging breakpoint that changes behavior.

```ruby
# bad
message = (1..9).map (&:to_s)
  .join (", ")

# good
message = (1..9).map(&:to_s).
  join(", ")
```

- Parentheses are optional when the method call exists entirely on one line.
- Implicit hashes are allowed.
- When an argument list spans multiple lines, break after the opening `(` and align arguments one per line, one indent level in from the opening line. Place the closing `)` at the beginning of a line, at the same indent level of the opening line.

```ruby
# bad
create_user(login,
            :email => email,
            :password => password
           )

# good
create_user(
  login,
  :email => email,
  :password => password
)

# good
create_user(login, {
  :email => email,
  :password => password
})

```

### Hash Literals

- Do not space pad hash literals to help visually distinguish them from single-line blocks.
- Both rockets `=>` and colons `:` are permitted. Do not mix the styles in the same hash.
- Space before and after the `=>`
- No trailing comma.
- When spanning multiple lines, break after the opening `{` and align elements one per line, one indent level in from the opening line. Place the closing `}` at the beginning of a line, at the same indent level of the opening line.

```ruby
# bad
digits = { "one"  =>    1,
           "two"  =>   22,
           "four" => 4444,
         }

# good
digits = {
  "one" => 1,
  "two" => 3,
  "four" => 4
}
```

### Array Literals

- Do not space pad arrays.
- Space after the comma. None before.
- No trailing comma.
- When spanning multiple lines, break after the opening `[` and align elements one per line, one indent level in from the opening line. Place the closing `]` at the beginning of a line, at the same indent level of the opening line.

```ruby
# bad
digits = [ 1,2,3 ]
letters = [ "doubleyou",
            "ecks",
            "why",
          ]

# good
digits = [1, 2, 3]
letters = [
  "doubleyou",
  "ecks",
  "why"
]
```
