# Overview

This document describes the API specification for the new Video-on-Demand (VOD) Upload service. For background regarding the VOD Upload product, refer to these documents:

* [Product Design](https://docs.google.com/document/d/1bxGQ-gPRyNOew1lM305zGncxPak2qDgQujrb4KBLOKk)
* [Tech Requirements](https://docs.google.com/document/d/1cIMAQJ0Bfi1fhs_WfNVRbKO30GwCLjduWDiEBGvcgzk)
* [Tech Spec (Outdated)](https://docs.google.com/document/d/1alGG0UB3F9PJotB3jTCScWOGYPtmMSLGs_RQEVZjBb4)

# Operations

## Uploads
* [Create New VOD](#create-new-vod)
* [Renew Upload Token](#renew-upload-token)
* [Upload VOD Part](#upload-vod-part)
* [Complete VOD Upload](#complete-vod-upload)
* [Get Upload VOD Status](#get-upload-vod-status)

## Other
* [List VODs](#list-vods)
* [Delete VOD](#delete-vod)
* [Update VOD](#update-vod)
* [Get VOD](#get-vod)

# Design Principles

In order of priority...

1. API must be secure
2. API must be versioned for backwards compatibility
3. API must be robust
4. API must be consistent with the rest of the Twitch API
5. API must allow resuming from catastrophic failure (one where all state in the client is lost)
6. API must allow for scaling the backend horizontally
7. API for uploading large amounts of data must be a separate endpoint that is publicly routable
8. API must be useable by both the web client and command-line clients

# Notes

1. The Valkyrie endpoint will be SSL-enabled.
2. The Valkyrie endpoint will only be available in one AWS region initially, but is not precluded from being deployed to multiple regions.
3. We follow the same error message format as described [here](https://github.com/justintv/Twitch-API/blob/master/README.md#errors).
4. Authentication is performed by the web frontend and OAuth, described [here](https://github.com/justintv/Twitch-API/blob/master/authentication.md). The spec below does *not* include authentication parameters.
5. Events will be ingested into [trace](https://git.xarth.tv/release/trace) for debugging purposes.

# Specification

This spec has drawn inspiration from the [S3 Multipart Upload API](http://docs.aws.amazon.com/AmazonS3/latest/dev/mpuoverview.html). S3 is a robust and scalable service that has been running a successful API for many years. In many ways, the service we are trying to create is similar to S3. Following this model will also make using S3 as our backend simpler.

## Specification Format

```
HTTP/1.1 (GET|PUT|POST) <dns endpoint><path>
- header-key: header-value
- header-key: header-value
<request body>
\n
=> error code: error
=> error code: error
=> 200 success
- header-key: header-value
- header-key: header-value
<response body json>
```

## Create New VOD

The purpose of this API is to create a new, empty VOD. Vinyl is responsible for VOD metadata, therefore it makes sense to host this API in there. Furthermore, Kraken/Web already does OAuth authentication, which we would like to leverage.

Importantly, this Vinyl API will return a string named `uploadToken`, which is a [cartman](https://git.xarth.tv/web/cartman) token. Vinyl will embed the `totalParts` field in this token. This token will be used to validate requests to Valkyrie.

*Version 1*

```
HTTP/1.1 POST https://api.twitch.tv/kraken/videos?channel_id=xxxxx
- x-api-version: 3
- content-type: application/json
{
  title: String,
  game: String,
  ... [vod params]
}

=> 401: Unauthorized
=> 404: NotFound (api version != 4)
=> 503: ServiceUnavailable
=> 200:
- content-type: application/json
- x-request-id: UUID
{
  upload: {
    token: String,
    url: "https://upload.twitch.tv/upload/c111111",
  },
  video: {
    title: "",
    description: "",
    broadcast_id: null,
    status: "created",
    _id: "c111111",
    tag_list: "",
    recorded_at: "2000-01-01T00:00:00Z",
    viewable_at: "2000-01-01T00:00:00Z",
    game: null,
    length: null,
    preview: "http://x.y.z/image.jpg",
    url: "http://www.twitch.tv/twitch/c/111111",
    views: 0,
    broadcast_type: "upload",
    _links: {
      self: "https://api.twitch.tv/kraken/videos/c111111",
      channel: "https://api.twitch.tv/kraken/channels/some_channel"
    }
    channel: {
      name: "some_channel",
      display_name: "Some Channel"
    }
  }
}
```

## Upload VOD part

This API will solely be responsible for receiving the raw video data from the client and tracking ingest progress.

Source: Valkyrie

*Version 1*

```
HTTP/1.1 POST https://uploads.twitch.tv/upload/:video_id?part=1&upload_token=xxxxx
body []byte

=> 400: UploadAlreadyCompleted
=> 400: UploadFailed
=> 401: Unauthorized
=> 404: NotFound (UnknownVideoId)
=> 411: ContentLengthRequired
=> 500: InternalServerError
=> 200: Success
- x-request-id: UUID
```

## Complete VOD Upload

This API will be called when a client has finished uploading all the parts for a Video.

_Note:_ We want to figure out whether it makes sense to tell the user they are
missing parts. It is easy to determine in the case where parts are missing in
the middle, but not easy if the parts are missing at the end unless they tell
us how many parts there should be.

Source: Valkyrie

*Version 1*

```
HTTP/1.1 POST https://uploads.twitch.tv/upload/:video_id/complete?upload_token=xxxxx

=> 400: MissingParts
=> 401: Unauthorized
=> 404: NotFound (InvalidVideoId)
=> 500: InternalServerError
=> 200: Success
- x-request-id: UUID
```

## Renew Upload Token

In the case where the user's upload token is lost or has expired, it can be renewed for a given `video_id`

Source: Vinyl

*Version: 2*

```
HTTP/1.1 POST https://api.twitch.tv/kraken/videos/:video_id/upload_token
- x-api-version: 3
=> 400: InvalidVideoId
=> 401: Unauthorized
=> 500: InternalServerError
=> 200:
- content-type: application/json
- x-request-id: UUID
{
  upload: {
    token: String,
    url: "https://uploads.twitch.tv/upload/c111111",
  }
}
```

## Get VOD upload status

This API will help clients discover the current state of an upload in order to resume from failure.

The response data includes a field named `upload_status` that is an array of boolean values, describing which parts have successfully been uploaded.

Source: Valkyrie

*Version: 2*

```
HTTP/1.1 GET https://uploads.twitch.tv/upload/:video_id?upload_token=xxxxx

=> 401: Unauthorized
=> 404: NotFound (InvalidVideoId)
=> 500: InternalServerError
=> 200: Success
- content-type: application/json
- x-request-id: UUID
{
  upload: {
    token: String,
    url: "https://uploads.twitch.tv/upload/c111111",
    status: [Boolean],
  }
}
```

## List VODs

Show all VODs for a given channel.

See https://github.com/justintv/Twitch-API/blob/master/v3_resources/videos.md#get-channelschannelvideos

Source: Vinyl

*Version: 1*

```
HTTP/1.1 GET https://api.twitch.tv/kraken/channels/:channel_name/videos
- x-api-version: 3

=> 400: InvalidChannel
=> 401: Unauthorized
=> 500: InternalServerError
=> 200:
- content-type: application/json
- x-request-id: UUID
{
  _total: Integer,
  videos: [
    {
      title: String,
      description: String,
      ...
    }
  ]
}
```

## Delete VOD

This API call allows users to delete their created VODs.

Source: Vinyl

*Version: 1*

```
HTTP/1.1 DELETE https://api.twitch.tv/kraken/videos/:id
- x-api-version: 3
=> 400: InvalidVideoId
=> 401: Unauthorized
=> 500: InternalServerError
=> 200:
- x-request-id: UUID
```

## Update VOD

To give a name, description and other fields to a specific VOD.

The response structure will look equivalent to https://github.com/justintv/Twitch-API/blob/master/v3_resources/videos.md#get-videosid

Source: Vinyl

*Version: 1*

```
HTTP/1.1 PUT https://api.twitch.tv/kraken/videos/:video_id
- x-api-version: 3
- content-type: application/json
{
  title: String,
  description: String,
  game: String,
  ... (see other metadata in Create new VOD above)
}

=> 400: InvalidVideoId
=> 401: Unauthorized
=> 500: InternalServerError
=> 200:
- content-type: application/json
- x-request-id: UUID
{
  video: {
    title: String,
    description: String,
    tag_list: String,
    broadcast_type: "upload",
    status: "uploading",
    ... (other metadata)
  }
}
```

## Get VOD

Retrieve data about a specific VOD.

The response structure will look equivalent to https://github.com/justintv/Twitch-API/blob/master/v3_resources/videos.md#get-videosid

Source: Vinyl

_Note 1:_ We want to give some thought to how we hide the un-published statuses if the requester is not the owner of this video.

_Note 2:_ `status` field will be one of `created`, `uploading`, `transcoding`, `unpublished`, `published`.

*Version: 1*

```
HTTP/1.1 GET https://api.twitch.tv/kraken/videos/:video_id
- x-api-version: 3

=> 401: Unauthorized
=> 404: NotFound (InvalidVideoId)
=> 500: InternalServerError
=> 200:
- content-type: application/json
- x-request-id: UUID
{
  title: String,
  description: String,
  tag_list: String,
  broadcast_type: "upload",
  status: "transcoding",
  channel: {
    name: String,
    ...
  }
  ... (see other metadata in Create new VOD above)
}
```

# References

* [Kraken API Documentation](https://github.com/justintv/Twitch-API/blob/master/README.md)
* [Cartman](https://docs.google.com/document/d/1_spjDG1J_D80Z0Q1dTPGZSyDG88UeMnJpb4Y3KLpzMg)
