# Internationalization
Internationalization to you should mean writing code that can be translated
easily into the foreign languages we support. For a look at the
internationalization pipeline, [see
here](/web/i18n/internationalization_pipeline.md). For questions this document
doesn't cover, email [i18n@twitch.tv](mailto:i18n@twitch.tv).

A new I18n process has been made for Web-Client, please read more about it [here](https://twitchtv.atlassian.net/wiki/pages/viewpage.action?pageId=164206952). The below is not applicable to Web Client anymore.

##Overview
There are two purposes to the translation function.

1. Identify the set of strings that need to be translated (by static code analysis, looking for the translation function `_()` or `i18n()` and recording the argument).
1. While code is running, we need to swap out our English text with the translated text. This is done by those same translation functions.


We want to give the translators as much context as possible, so it's important
not to break up sentences, to use named placeholders for variables, and to keep
punctuation inside the string. Also, we want to be able to NOT translate some
things like legal text and internal tools. Specifics on how to do that below.


##Coding with JavaScript
We identify strings in javascript by regex matching on .js files
for the i18n function.

The translation function in JavaScript is i18n because of the wonderful
underscore library. It's basically just a single language hash that's passed
into ember from rails. The simplest use is just:

```javascript
var translatedString = i18n("English string");
```

It's generally a bad idea to call i18n on variables instead of just string
literals because it messes with the static analysis that sends the strings to
our vendor to be translated.

```javascript
var messageBox = outbox ? "Outbox" : "Inbox";
var displayString = i18n(messageBox); // BAD BAD BAD.
```

There sadly isn't any string interpolation native to JavaScript, so we use `{{ }}` style
interpolation with the Mustache library. Make sure to apply the interpolation
AFTER the `i18n`. This will allow the translator to understand what the
placeholder means and move it around logically for the target language:

```javascript
var translatedFormat = i18n("welcome to {{userName}}'s broadcast!");
var translatedStr = Mustache.render(translatedFormat, {userName: $('#user').text()});
```

If you want to NOT translate a string (maybe it's legal text or just for an
internal tool), just don't use the `i18n` function.


##Coding with Ruby templates
We identify strings in Rails by precompiling the haml templates and finding the
arguments to the translation function.

The translation function in Rails is `_()`. However, you can translate strings in
haml without explicitly calling `_()`. We use the [Magic Translations
gem](https://git.xarth.tv/web/haml-magic-translations) to
automagically wrap all free text in haml with the underscore function. This is
good because it allows us to be lazy sometimes and have cleaner code. It's bad
because we can translate things we don't want to and not realize it. CMS emails
are haml, so same deal.

This simplest uses are just (all equivalent):

```ruby
%p Top PS4 Channels #preferred
%p
  Top PS4 Channels
%p
  = _("Top PS4 Channels")
```

It's generally a bad idea to call `_()` on variables instead of just string
literals because it messes with the static analysis that sends the strings to
our vendor to be translated:

```ruby
message_boxes = ["inbox", "outbox", "other"]
message_box_strings = message_boxes.map {|s| _(s)} # tempting, but BAD BAD BAD.
```

If you want to use interpolation, use `%{varname}` style interpolation with
meaningful and not-necessarily-a-tech-person readable variable names. Never use
string `#{}` style string interpolation. Make sure to apply the interpolation
AFTER the translation:

```ruby
= span _("Log In and Follow %{channel}") % {:channel => params[:follow]}
```

One of the worst thing that you can do is remove context from a string. Here's
an example of doing this WRONG:

```ruby
%p.info
  = channel.channel_count
  = _("viewers on")
  = channel.channel_name #BAD BAD BAD
```

That gives the translator no context and no ability to move the subject and the
count around, as is necessary in some other languages. Instead, do it this way:

```ruby
%p.info
  = _("%{num_viewers} viewers on %{channel_name}") % {:num_viewers => channel.channel_count, :channel_name => channel.channel_name}
```

If you want to wrap html around a part of a string, DO NOT just break up the
string and add the html in between, instead, use the helper method `l_`, defined
in lib/translations.rb. We don't want our translators having to mess with that.
They know about the $$ syntax:

```ruby
#setup_link
 = l_ :link_to, _("First time broadcasting? Check out the $Streaming Apps$", '/broadcast'
```
which generates:

```html
<div id="setup_link">
  First time broadcasting? Check out the <a href="/broadcast">Streaming Apps</a>
</div>
```

And if you have multiple per string, use lx_. It is a little bit dangerous,
however, as the order of the $$s might change in some languages if your string
is syntactically complicated. Use caution:

```ruby
%p.form_microcopy
  = lx_([:link_to, :link_to], _("To enable Two-factor authentication, please verify your $email$ and $phone number.$"), [{:section => 'profile'}, {:section => 'notifications'}])
```

You can use `l_` and `lx_`  for `:button`, `:strong`, `:em`, `:blockquote`, `:span`, `:codetag`,
`:p`, `:div`, `:b`, `:link_to`, ...


If you want to NOT translate a string (maybe it's legal text or just for an
internal tool), use the_no_translation function:

```ruby
%h1 = _no_translation("DMCA Takedown")
%h1 = "DMCA Takedown" # this is functionally equivalent, but unclear that it's purpose is to not translate the string. prefer the former.
```

##Coding with Liquid in CMS Emails
See https://github.com/Shopify/liquid/wiki/Liquid-for-Designers for a basic guide on Liquid's syntax. This section should explain all the different types of i18n you want to in Liquid CMS Email templates. Besides the functions defined by liquid, we have also defined helper functions in /lib/liquid_tags.rb

**1. Basic i18n usage:**

i18ning basic strings in Liquid is easy, it looks like:

```liquid
{% translate %}This is my good string to be translated by Golf{% endtranslate %}
```

We just pass the string to our `translate` function. `translate` is a function that takes the passed in string and translates it.

**2. i18n with string interpolation:**

i18ning strings that require string interpolation just as easy. The syntax looks like:

```liquid
{% translate adjective: "good", user: user %}This is my %{adjective} string to be translated by %{user}{% endtranslate %}
```

The `translate` function optionally takes a hash of strings to interpolate into the given i18n string. The values in this hash can either be hardcoded strings like `"good"` or they can refer to a variable name like `user`.

**3. Links with `twitch_uid_href` (and passing correct params to Mixpanel)**

```liquid
  <a
      href="{{ '/profile' | twitch_uid_href: recipient: recipient_name, location: 'action' }}"
      style="background-color:#6441a5;color:#ffffff;">
    {% translate %}View your profile{% endtranslate %}
  </a>
```

`twitch_uid_href` is a function that takes in the string to use as the href, then takes in an optional list of arguments to pass to Mixpanel. Note that Liquid's hash syntax is a bit weird. In the example above, the resulting hash passed to Mixpanel is `{recipient: recipient_name, location: 'action'}`.

If the URL is more complicated than just a hardcoded string, you may need to perform some logic on the URL before passing it to `twitch_uid_href`. Example:

```liquid
  {% assign verification_url = '/user/verify_email?email_verification_code=' | append: email_verification_code %}

  <a
      href="{{ verification_url | twitch_uid_href: recipient: recipient_name, location: 'action' }}"
      style="background-color:#6441a5;color:#ffffff;">
    {% translate %}Verify your email{% endtranslate %}
  </a>
```

---

**4. `format` for calling functions in $dollar sign delimited$ i18n strings.**

If you have a i18n string like 'Hey Twitch user, $you're awesome!$' and you want to call a function on the substring in the dollar signs, you would do:

```liquid
{% format type: "strong" %}
  {% translate %}Hey Twitch user, $you're awesome!${% endtranslate %}
{% endformat %}
```

`format` is a function that takes in an i18n string with `$` signs then performs a given function on that substring. Here we first pass the string to be translated to `translate`, then that translated string is passed into `format` to generate the correctly translated HTML like:

```html
Hey Twitch user, <strong>you're awesome!</strong>
```

`format` takes additional hash pairs to pass to the given function. To apply styles to the generated tag, you can do:

```liquid
{% format type: "strong", style: "color:#999", id: "awesome" %}
  {% translate %}Hey Twitch user, $you're awesome!${% endtranslate %}
{% endformat %}
```

This also works with variables:

```liquid
{% format type: "strong", style: "color:#999", id: "awesome" %}
  {% translate username: "golftest30" %}Hey %{username}, $you're awesome!${% endtranslate %}
{% endformat %}
```

"strong", "span", "link_to", "twitch_remove_link" are the only `type`s that are allowed to be used with l_

**5. Chaining `format` calls**

To process a string with multiple dollar sign blocks, chain calls to `format` like:

```liquid
{% format type: "link_to", href: "http://blog.twitch.tv/" %}
  {% format type: "link_to", href: "https://www.facebook.com/Twitch" %}
    {% format type: "link_to", href: "http://www.twitter.com/twitch" %}
      {% translate %}Be sure to Follow Twitch on $Twitter$ and $Facebook$, and check out the $Official Twitch Blog${% endtranslate %}
    {% endformat %}
  {% endformat %}
{% endformat %}
```
