# Sudo tokens
Our web services have a concept of a "sudo mode" -- when a user needs to access or change private information,
like changing an email address, password, or partners filling out tax documents, we require that they elevate
into a sudo-like mode by reentering their credentials. Each time a user enters their credentials, they are given
elevated permissions to do these sorts of things for one hour.

## Requiring sudo mode for your feature
If you wish to gate a feature behind sudo mode, include the `Sessions` module in your controller and add a
`require_sudo` call as a `before_filter`, like so:

```ruby
class RocketGogglesController < ApplicationController
  include Sessions
  
  before_filter :require_sudo
  
  def index
    @rocket_goggles = RocketGoggle.all
  end
end
```

The ability to see rocket goggles is now gated behind sudo mode. If a user tries to access this controller
action and they haven't logged in on their current machine in the last hour, they'll be redirected to a login
page, and redirected back to the rocket goggles index page when they've successfully logged in.

## How it works
When a user's session is created, in addition to normal session cookies we give the user a 1-hour, HTTP-only,
secure cookie called `sudo`. The cookie contains a JWT generated by the `Sessions::GenerateSudoToken` interactor.

If this cookie doesn't exist when we run into a `require_sudo` call, we log the user out and direct them back to
the log in page so that they can receive one. We log the user out because there's not currently a
clean way to log in while already logged in, so until Passport gains that functionality, a logout is forced.

## Testing in development
Because the cookie we give the user is marked [secure][1], it will not be transmitted over HTTP (only HTTPS).
And because we do not have a fully vetted way to run web/web under HTTPS locally, this cookie will not be transmitted
when operating locally. Therefore, all attempts to gain sudo mode will (technically succeed but) fail to be recognized
by features that require it.

As a temporary fix to this, you can comment out the line that makes this cookie secure, currently in
`PassportController#callback`. It looks like `secure: true,`. However, this is a bad solution to enable sudo mode
locally, and we want something less risky and more thorough (be sure not to commit that change). Please feel free
to come up with a better solution than this one!!

[1]: https://en.wikipedia.org/wiki/HTTP_cookie#Secure_and_HttpOnly
