# Version

0.1

# Example pipeline workflow

## Setup

This assumes you have a terraform environment built with https://git.xarth.tv/twitch/terraform and a common makefile you import similar to https://git.xarth.tv/feeds/service-common/blob/master/CommonMakefile.mk

## Make a local change

Make your change locally on your machine.

```sh
vim ./cmd/service/main.go
```

## Try your change locally

To run your code locally

```sh
make run
```

You should see output similar to this
```sh
> make run
env ENVIRONMENT=development go run -race ./cmd/duplo/main.go
msg="Debug logging turned on"
level=debug msg="Debug log setup complete"
addr=graphite.internal.justin.tv:8125 prefix=feeds.duplo.development.f45c89cadf27_ant_amazon_com msg="Statsd setup complete"
...
```

## (Optional) See your change in the real staging cluster

This step requires you install [Docker for mac](https://docs.docker.com/docker-for-mac/install/).

To see your change in a real cluster, you can deploy that single commit to staging.  First, commit your change.

```sh
git commit -am 'This is my change'
```

Then deploy it to the remote cluster

```sh
make local_deploy
```

You should see output similar to
```
< make local_deploy
docker run -w /go/src/code.justin.tv/feeds/duplo -v `pwd`:/go/src/code.justin.tv/feeds/duplo docker.pkgs.xarth.tv/feeds/gobuilder:1.8.3-1 make static_build
go build -ldflags "-linkmode external -extldflags -static" -o ./cmd/duplo/duplo ./cmd/duplo
go build -ldflags "-linkmode external -extldflags -static" -race -o ./cmd/duplo/duplo_race ./cmd/duplo
...
```

This will build a Docker image locally, push the image, build a task definition, and deploy the task definition to the staging cluster.  Towards the end of your output, you should something similar to this

```sh
Sleeping till task not in PENDING state...
Container: duplo
tcp 0.0.0.0:6060 => 10.198.42.40:32902
tcp 0.0.0.0:8000 => 10.198.42.40:32901
Adding to target group arn:aws:elasticloadbalancing:us-west-2:724951484461:targetgroup/feeds-duplo-staging/d2f515e672c48fff value 10.198.42.40:32901
Waiting for Ctrl+C
--------------------------------------------------------------------------------
msg="Debug logging turned off"
addr=ip-10-198-42-40.us-west-2.compute.internal:8125 prefix=feeds.duplo.staging.i-08a524885f7bb7f28 msg="Statsd setup complete"
code_version=1483d837a2bf42f5876afd84b3d275894e87818b-jlindamo pid=1 msg="Common setup completed"
service_count=4 msg="Starting service execution"
```

Here it is telling you which ports you can send traffic to, as well as pipe the instance's stdout to your console.  To stop your ad-hoc deploy, type Ctrl+C.  This should cleanly shutdown your service.  You will see output similar to this.

```sh
^CKill signal observed
Waiting for stdout stream to stop
stdout stream stopped
Trying to StopTask
StopTask finished
Trying to clean up freshly created task definition
DeregisterTaskDefinition finished
```

You can then clean up any artifacts created by this local deployment inside your repository.

```sh
< make clean
rm -f ./cmd/duplo/duplo_race ./cmd/duplo/duplo test_report.xml
```

## Run tests

You can run unit tests locally with `make test`.  The output will look similar to this.

```sh
> make test
go test -v -race "./cmd/..." | go-junit-report -set-exit-code > test_report.xml
```

You can run integration tests locally with `make integration_test`.  The output will look similar to this.

```sh
< make integration_test
env ENVIRONMENT=development go test -race -run=. -tags=integration "./cmd/..."
ok  	code.justin.tv/feeds/duplo/cmd/duplo	23.239s
?   	code.justin.tv/feeds/duplo/cmd/duplo/internal/activity	[no test files]
?   	code.justin.tv/feeds/duplo/cmd/duplo/internal/api	[no test files]
?   	code.justin.tv/feeds/duplo/cmd/duplo/internal/api/mocks	[no test files]
ok  	code.justin.tv/feeds/duplo/cmd/duplo/internal/db	1.041s
ok  	code.justin.tv/feeds/duplo/cmd/duplo/internal/db/dupdynamo	1.112s
```

## (Optional) Vendor new dependencies

If you added a new library or updated a dependency, you should trigger a fresh `dep` update to pick up that dependency.

```sh
make revendor
```

The output will look something like this.

```sh
< make revendor
rm -rf ./vendor
dep ensure
dep prune
# Do helpers
mkdir -p vendor/code.justin.tv/twitch/ecs-deploy/helpers/
cd ../../twitch/ecs-deploy && git fetch -av
remote: Counting objects: 3, done.
remote: Compressing objects: 100% (3/3), done.
remote: Total 3 (delta 0), reused 0 (delta 0), pack-reused 0
...
```

## Format your code

You should format your code before pushing it for a pull request.

```sh
make fix
```

The output should look something like this.

```sh
< make fix
find . -iname '*.go' -not -path '*/vendor/*' -print0 | xargs -0 gofmt -s -w
find . -iname '*.go' -not -path '*/vendor/*' -print0 | xargs -0 goimports -w
terraform fmt
```

## (Optional) Run lints

You can optionally run lints locally to detect lint errors before you push.  This step sometimes takes a long time, so it's OK to just let Jenkins run these tests for you after you push if you like.

```sh
make lint
```

The output should look something like this.

```sh
< make lint
gometalinter --vendor --dupl-threshold=150 --min-confidence=.3 --tests --deadline=90s --disable-all -Egolint -Etest -Eineffassign -Etestify -Eunconvert -Estaticcheck -Egoconst -Egocyclo -Eerrcheck -Egofmt -Evet -Edupl -Einterfacer -Estructcheck -Evetshadow -Egosimple -Egoimports -Evarcheck -Emisspell -Ealigncheck -Etest --exclude=/usr/local/go/src --exclude='/mocks/[^/]+\.go.*\((dupl)|(golint)\)' ./...
```

## Push your change

When ready, push your change to github.

```sh
git push origin HEAD:a_fix
```

## Point people to your ad-hoc deploy.

You can use the ecs-on-demand service to give reviewers a URL they can access to test your code.  The URL will be similar to http://1483d837a2bf42f5876afd84b3d275894e87818b.feeds-duplo.ecs-on-demand.us-west2.justin.tv.  You replace the GIT_SHA with the GIT_SHA of your push and `feeds-duplo` with the name of your service (for example `feeds-masonry`).

*Be Patient* The first access of this URL will be slow (around 10 seconds) while it deploys your commit.  After that, the URL should load quickly.

## Merge your code

When ready, merge your code with github.  When the merge to master happens, master should build and trigger a pipeline for your deployment.  Your team will have a pipeline deployment page where you can see all pipelines.

| Team | Pipeline page |
|------|-------------------|
| Feeds | https://jenkins-og.xarth.tv/view/feeds/ |


## Go to the pipeline deployment for your service

Each service will have a pipeline deployment.  This is usually ` integration => staging => canary => production `.  You can find your deployment pipeline from your team's Jenkins view.  For example, here are some feeds deployment pipelines.

| Service | Pipeline page |
|---------|---------------|
| Duplo   | https://jenkins-og.xarth.tv/view/feeds/job/feeds-duplo-pipeline/ |

## Watch your service deploy to staging

You can watch your service auto deploy to staging or integration by opening the deployment job box on the pipeline.  You will link to a page similar to this one for feeds-duplo-integration.

https://jenkins-og.xarth.tv/job/feeds-duplo-integration/34/console

The Jenkin's job output will include a task's STDOUT as well as debugging information for the deployment.

## Verify staging works

Visit the rollbar and grafana pages of your product and verify there are no new errors and metrics look ok.  Here are some example rollbar and grafana pages.

| Service | Rollbar | Grafana |
|---------|---------|---------|
| Duplo | https://rollbar.com/Twitch/Duplo/ | https://grafana.internal.justin.tv/dashboard/db/duplo?refresh=1m&orgId=1 |

## Manually promote canary

If integration and staging build and push correctly, the pipeline will wait for manual input before pushing canary.  Click "Accept" on the Pipeline page's Canary box.  This will start the canary deployment.

## Verify canary works

After canary has fully deployed, check graphs and rollbar for your service to ensure canary has deployed correctly and nothing broke.

Canary gets real, production traffic and is behind the production ALB.  You can even send it traffic directly to test responses.

## Push production

If canary works, click the "Accept" button on the pipeline's Production box to push to production.

## (Optional) Emergency push production

Sometimes emergencies happen and you need to quick push a commit to an environment.  To do this, go directly to the pipeline build for your environment and click "Build with Parameters" and insert the git SHA that you want to deploy.  Here are some examples:

| Service | Environment | Pipeline page |
|---------|---------------|-------------|
| Duplo   | Production | https://jenkins-og.xarth.tv/job/feeds-duplo-production/ |
