package main

import (
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatchlogs"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/ecs"
	"github.com/aws/aws-sdk-go/service/elbv2"
	"github.com/aws/aws-sdk-go/service/iam"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/aws/aws-sdk-go/service/sts"
)

type cachedSession struct {
	err     error
	session *session.Session
}

type awsapi struct {
	sessions    map[string]*cachedSession
	coreConfigs []*aws.Config
}

func (d *awsapi) getIAMClient(region string, awsRole string, profile string) (*iam.IAM, error) {
	awsSession, err := d.getAWSSession(region, "", profile)
	if err != nil {
		return nil, err
	}
	iamClient := iam.New(awsSession)
	return iamClient, nil
}

func (d *awsapi) getECSClient(region string, awsRole string, profile string) (*ecs.ECS, error) {
	awsSession, err := d.getAWSSession(region, awsRole, profile)
	if err != nil {
		return nil, err
	}
	ecsClient := ecs.New(awsSession)
	return ecsClient, nil
}

func (d *awsapi) getEC2Client(region string, awsRole string, profile string) (*ec2.EC2, error) {
	awsSession, err := d.getAWSSession(region, awsRole, profile)
	if err != nil {
		return nil, err
	}
	ec2Client := ec2.New(awsSession)
	return ec2Client, nil
}

func (d *awsapi) getELBClient(region string, awsRole string, profile string) (*elbv2.ELBV2, error) {
	awsSession, err := d.getAWSSession(region, awsRole, profile)
	if err != nil {
		return nil, err
	}
	elbClient := elbv2.New(awsSession)
	return elbClient, nil
}

func (d *awsapi) getS3Client(region string, awsRole string, profile string) (*s3.S3, error) {
	awsSession, err := d.getAWSSession(region, awsRole, profile)
	if err != nil {
		return nil, err
	}
	s3client := s3.New(awsSession)
	return s3client, nil
}

func (d *awsapi) getSSMClient(region string, awsRole string, profile string) (*ssm.SSM, error) {
	awsSession, err := d.getAWSSession(region, awsRole, profile)
	if err != nil {
		return nil, err
	}
	ssmClient := ssm.New(awsSession)
	return ssmClient, nil
}

func (d *awsapi) getLogClient(region string, awsRole string, profile string) (*cloudwatchlogs.CloudWatchLogs, error) {
	awsSession, err := d.getAWSSession(region, awsRole, profile)
	if err != nil {
		return nil, err
	}
	cloudwatchClient := cloudwatchlogs.New(awsSession)
	return cloudwatchClient, nil
}

func (d *awsapi) getAWSSession(region string, awsRole string, profile string) (*session.Session, error) {
	if d.sessions == nil {
		d.sessions = map[string]*cachedSession{}
	}
	key := region + "_" + awsRole + "_" + profile
	if prev, exists := d.sessions[key]; exists {
		return prev.session, prev.err
	}
	theseConfigs := append(make([]*aws.Config, 0, len(d.coreConfigs)+1), d.coreConfigs...)
	theseConfigs = append(theseConfigs, &aws.Config{
		Region: &region,
	})
	stored := cachedSession{}
	d.sessions[key] = &stored
	return stored.create(region, awsRole, profile, theseConfigs)
}

func (c *cachedSession) create(region string, awsRole string, profile string, coreConfigs []*aws.Config) (*session.Session, error) {
	if c.session == nil && c.err == nil {
		c.session, c.err = func() (*session.Session, error) {
			opts := session.Options{
				Config: aws.Config{
					Region:              aws.String(region),
					STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
				},
				SharedConfigState: session.SharedConfigEnable,
				Profile:           profile,
			}
			opts.Config.MergeIn(coreConfigs...)
			awsSessionForRole, err := session.NewSessionWithOptions(opts)
			if err != nil {
				return nil, err
			}
			if awsRole == "" {
				return awsSessionForRole, nil
			}
			stsclient := sts.New(awsSessionForRole)
			arp := &stscreds.AssumeRoleProvider{
				ExpiryWindow: 10 * time.Second,
				RoleARN:      awsRole,
				Client:       stsclient,
			}
			credsARP := credentials.NewCredentials(arp)
			theseConfigs := append(make([]*aws.Config, 0, len(coreConfigs)+1), coreConfigs...)
			theseConfigs = append(theseConfigs, &aws.Config{
				Credentials:         credsARP,
				Region:              aws.String(region),
				STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
			})
			return session.NewSession(theseConfigs...)
		}()
	}
	return c.session, c.err
}
