package main

import (
	"encoding/json"
	"sync"

	"fmt"

	"code.justin.tv/feeds/errors"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/hashicorp/consul/api"
	"github.com/hashicorp/terraform/terraform"
	"github.com/urfave/cli"
	"golang.org/x/sync/errgroup"
)

type consulHelper struct {
	onceKv sync.Once
	kv     *api.KV
	kvErr  error
}

func (d *consulHelper) lsDir(dir string) ([]string, error) {
	kv, err := d.createConsulClient()
	if err != nil {
		return nil, err
	}
	pairs, _, err := kv.Keys("/"+dir+"/", "/", nil)
	if err != nil {
		return nil, err
	}
	return uniq(trimmeKeys([]string{dir + "/", dir}, pairs)), nil
}

func (d *consulHelper) createConsulClient() (*api.KV, error) {
	d.onceKv.Do(func() {
		config := api.DefaultConfig()
		config.Address = "api.us-west-2.prod.consul.live-video.a2z.com"
		config.Datacenter = "us-west2"

		client, err := api.NewClient(config)
		if err != nil {
			d.kvErr = err
			return
		}
		d.kv = client.KV()
	})
	return d.kv, d.kvErr
}

func (d *consulHelper) teamServices(team string) ([]string, error) {
	return d.lsDir("pipeline/deployed/" + team)
}

func (d *consulHelper) teams() ([]string, error) {
	return d.lsDir("pipeline/info")
}

func (d *consulHelper) serviceEnvs(team string, service string) ([]string, error) {
	return sortEnvs(d.lsDir("pipeline/deployed/" + team + "/" + service + "/"))
}

func (d *consulHelper) infoServiceEnvs(team string, service string) ([]string, error) {
	return sortEnvs(d.lsDir("pipeline/info/" + team + "/" + service + "/"))
}

func (d *consulHelper) services() ([]string, error) {
	teams, err := d.teams()
	if err != nil {
		return nil, err
	}
	var mu sync.Mutex
	s := make([]string, 0, len(teams)*10)
	eg := errgroup.Group{}
	for _, team := range teams {
		team := team
		eg.Go(func() error {
			svcs, err := d.teamServices(team)
			if err != nil {
				return err
			}
			mu.Lock()
			for _, svc := range svcs {
				s = append(s, team+"-"+svc)
			}
			mu.Unlock()
			return nil
		})
	}
	return s, eg.Wait()
}

func (d *consulHelper) deployedVersion(team string, service string, env string) (string, error) {
	kv, err := d.createConsulClient()
	if err != nil {
		return "", err
	}
	key := "pipeline/deployed/" + team + "/" + service + "/" + env
	pair, _, err := kv.Get(key, nil)
	if err != nil {
		return "", err
	}
	if pair == nil {
		return "", nil
	}
	return string(pair.Value), nil
}

func (d *consulHelper) setDeployedVersion(team string, service string, env string, tag string) error {
	kv, err := d.createConsulClient()
	if err != nil {
		return err
	}
	kvpair := &api.KVPair{
		Key:   fmt.Sprintf("pipeline/deployed/%s/%s/%s", team, service, env),
		Value: []byte(tag),
	}
	_, err = kv.Put(kvpair, nil)
	return err
}

type deploymentInfo struct {
	DeployAWSRole   string                 `json:"deploy_aws_role,omitempty"`
	AWSCreds        string                 `json:"aws_creds,omitempty"`
	Region          string                 `json:"region,omitempty"`
	PromoteFrom     string                 `json:"promote_from,omitempty"`
	AutoPromote     bool                   `json:"auto_promote,omitempty"`
	Profile         string                 `json:"profile,omitempty"`
	TerraformStates []terraformStateRemote `json:"terraform_states,omitempty"`
	TaskTemplate    string                 `json:"task_template,omitempty"`
	Data            map[string]string      `json:"data,omitempty"`
	ClusterName     string                 `json:"cluster_name,omitempty"`
	ServiceName     string                 `json:"service_name,omitempty"`
}

func (d *deploymentInfo) getRole(c *cli.Context) string {
	roleToUse := d.DeployAWSRole
	if c.GlobalBool("skiprole") || c.Bool("skiprole") {
		roleToUse = ""
	}
	return roleToUse
}

func (d *deploymentInfo) readTerraformState(c *cli.Context, awsapi *awsapi) (map[string]string, error) {
	ret := make(map[string]string, 24)
	for _, s := range d.TerraformStates {
		roleToUse := d.getRole(c)
		s3Client, err := awsapi.getS3Client(s.Region, roleToUse, d.Profile)
		if err != nil {
			return nil, err
		}
		state, err := s._readTerraformState(s3Client)
		if err != nil {
			return nil, err
		}
		for k, v := range state {
			ret[k] = v
		}
	}
	for k, v := range d.Data {
		ret[k] = v
	}
	return ret, nil
}

type terraformStateRemote struct {
	Region      string `json:"region,omitempty"`
	Bucket      string `json:"bucket,omitempty"`
	Key         string `json:"key,omitempty"`
	cachedState map[string]string
}

func (t *terraformStateRemote) _readTerraformState(s3Client *s3.S3) (map[string]string, error) {
	if t.cachedState != nil {
		return t.cachedState, nil
	}
	out, err := s3Client.GetObject(&s3.GetObjectInput{
		Bucket: &t.Bucket,
		Key:    &t.Key,
	})
	if err != nil {
		return nil, err
	}
	state := terraform.State{}
	if err := json.NewDecoder(out.Body).Decode(&state); err != nil {
		return nil, err
	}
	terraformOutputs := make(map[string]string, len(state.RootModule().Outputs))
	for k, v := range state.RootModule().Outputs {
		terraformOutputs[k], _ = v.Value.(string)
	}
	t.cachedState = terraformOutputs
	return terraformOutputs, nil
}

func (d *consulHelper) getActualDeploymentInfo(team string, service string, env string) (*deploymentInfo, error) {
	return d.getDeploymentInfo(team, service, env)
}

func (d *consulHelper) getDeploymentInfo(team string, service string, env string) (*deploymentInfo, error) {
	kv, err := d.createConsulClient()
	if err != nil {
		return nil, err
	}
	key := "pipeline/info/" + team + "/" + service + "/" + env
	p, _, err := kv.Get(key, nil)
	if err != nil {
		return nil, err
	}
	if p == nil {
		return nil, errors.Errorf("unable to find consul key %s", key)
	}
	var ret deploymentInfo
	if err := json.Unmarshal(p.Value, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}
