package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"strings"
	"text/template"

	"code.justin.tv/feeds/errors"
	"github.com/aws/aws-sdk-go/service/ecs"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/fatih/color"
	"github.com/urfave/cli"
)

type ecsCommand struct {
	completes    *completes
	consulHelper *consulHelper
	awsapi       *awsapi
	logger       *logger
}

type statusResult struct {
	Team               string
	Service            string
	Env                string
	TerraformState     map[string]string
	ECSService         *ecs.Service
	DeployedVersion    string
	PromoteFrom        string
	PromoteFromVersion string
	Short              bool
}

func (s *statusResult) fullTemplate(out io.Writer) error {
	return fullStatusTemplate.Execute(out, s)
}

func (s *statusResult) ConsulStatus() string {
	if len(s.ECSService.Deployments) != 1 {
		return color.RedString("Being deployed")
	}
	if s.DeployedVersion != s.PromoteFromVersion {
		return color.YellowString("Undeployed")
	}

	return color.GreenString("OK")
}

func (s *statusResult) Center(text string) string {
	leftOver := 80 - len(text)
	if leftOver <= 0 {
		return text
	}
	return strings.Repeat("=", leftOver/2) + text + strings.Repeat("=", (leftOver+1)/2)
}

func (d *ecsCommand) subcommands() cli.Command {
	return cli.Command{
		Name:  "ecs",
		Usage: "ECS helper commands",
		Flags: []cli.Flag{
			cli.BoolFlag{
				Name:  "skiprole, r",
				Usage: "If set, will skip elevating aws roles during deployments.",
			},
		},
		Subcommands: []cli.Command{
			{
				Name:         "block-till-stable",
				Usage:        "Block till deployment is stable",
				ArgsUsage:    "[team] [service] [environment]",
				Action:       errExit(d.blockTillStable),
				BashComplete: d.statusComplete,
			},
			{
				Name:         "status",
				Usage:        "Show the status of a full deployment pipeline",
				ArgsUsage:    "[team] [service] (environment)",
				Action:       errExit(d.statusAction),
				BashComplete: d.statusComplete,
			},
		},
	}
}

func (d *ecsCommand) statusComplete(c *cli.Context) {
	d.completes.teamServiceEnvComplete(c)
	fmt.Fprintln(c.App.Writer, "")
}

func (d *ecsCommand) blockTillStable(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)
	deploymentInfo, err := d.consulHelper.getActualDeploymentInfo(team, service, env)
	if err != nil {
		return err
	}
	roleToUse := deploymentInfo.getRole(c)
	ecsClient, err := d.awsapi.getECSClient(deploymentInfo.Region, roleToUse, deploymentInfo.Profile)
	if err != nil {
		return err
	}

	clusterName := deploymentInfo.ClusterName
	serviceName := deploymentInfo.ServiceName
	reqInfo := ecs.DescribeServicesInput{
		Cluster:  &clusterName,
		Services: []*string{&serviceName},
	}
	err = ecsClient.WaitUntilServicesStable(&reqInfo)
	if err != nil {
		return err
	}
	fmt.Fprintln(c.App.Writer, "OK")
	return nil
}

func (d *ecsCommand) createTaskDefinitionInput(c *cli.Context, deploymentInfo *deploymentInfo) (*ecs.RegisterTaskDefinitionInput, error) {
	templateParams := map[string]interface{}{}

	containerTemplate := deploymentInfo.TaskTemplate
	if containerTemplate == "" {
		containerTemplate = defaultTaskTemplateText
	}

	state, err3 := deploymentInfo.readTerraformState(c, d.awsapi)
	if err3 != nil {
		return nil, err3
	}
	for k, v := range state {
		templateParams[k] = v
	}

	tmplate, err2 := template.New("containers").Parse(containerTemplate)
	if err2 != nil {
		return nil, errors.Wrapf(err2, "Unable to parse containers template %s", containerTemplate)
	}
	buf := bytes.Buffer{}
	if err := tmplate.Execute(&buf, templateParams); err != nil {
		return nil, errors.Wrap(err, "Unable to process template")
	}

	abc := ecs.RegisterTaskDefinitionInput{}
	if err := json.NewDecoder(&buf).Decode(&abc); err != nil {
		return nil, errors.Wrap(err, "Unable to read container template file as containers")
	}
	return &abc, nil
}

func (d *ecsCommand) statusAction(c *cli.Context) error {
	if c.NArg() > 3 || c.NArg() < 2 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	if c.NArg() == 2 {
		envs, err := d.consulHelper.serviceEnvs(team, service)
		if err != nil {
			return cli.NewExitError(err.Error(), 1)
		}
		for _, env := range envs {
			if env == "latest" {
				continue
			}
			if err := d.statusActionForEnv(c, team, service, env, true); err != nil {
				return err
			}
		}
		return nil
	}
	env := c.Args().Get(2)
	return d.statusActionForEnv(c, team, service, env, false)
}

func (d *ecsCommand) statusActionForEnv(c *cli.Context, team string, service string, env string, short bool) error {
	version, err := d.consulHelper.deployedVersion(team, service, env)
	if err != nil {
		return err
	}
	deploymentInfo, err := d.consulHelper.getDeploymentInfo(team, service, env)
	if err != nil {
		return err
	}
	roleToUse := deploymentInfo.getRole(c)
	if err2 := d.logger.verboseLogYourself(c, deploymentInfo); err2 != nil {
		return err2
	}
	currentState, err := deploymentInfo.readTerraformState(c, d.awsapi)
	if err != nil {
		return err
	}
	clusterName := deploymentInfo.ClusterName
	serviceName := deploymentInfo.ServiceName

	ecsService, err := d.discoverDeployedService(clusterName, serviceName, roleToUse, deploymentInfo)
	if err != nil {
		return err
	}
	s := statusResult{
		Team:            team,
		Service:         service,
		Env:             env,
		TerraformState:  currentState,
		ECSService:      ecsService,
		DeployedVersion: version,
		PromoteFrom:     deploymentInfo.PromoteFrom,
		Short:           short,
	}
	if deploymentInfo.PromoteFrom != "" {
		fromVersion, err := d.consulHelper.deployedVersion(team, service, deploymentInfo.PromoteFrom)
		if err != nil {
			return err
		}
		s.PromoteFromVersion = fromVersion
	}
	return s.fullTemplate(c.App.Writer)
}

func (d *ecsCommand) readTerraformState(s3client *s3.S3, bucket string, key string) (map[string]string, error) {
	return readTerraformState(s3client, bucket, key)
}

func (d *ecsCommand) discoverDeployedService(clusterName string, serviceName string, roleToUse string, deploymentInfo *deploymentInfo) (*ecs.Service, error) {
	describeInput := &ecs.DescribeServicesInput{
		Cluster:  &clusterName,
		Services: []*string{&serviceName},
	}
	ecsClient, err := d.awsapi.getECSClient(deploymentInfo.Region, roleToUse, deploymentInfo.Profile)
	if err != nil {
		return nil, err
	}
	servicesState, err := ecsClient.DescribeServices(describeInput)
	if err != nil {
		return nil, err
	}
	if len(servicesState.Failures) > 0 {
		return nil, errors.Errorf("service description failure: %s", *servicesState.Failures[0].Reason)
	}
	if len(servicesState.Services) == 0 {
		return nil, errors.New("unable to find service")
	}
	if len(servicesState.Services) != 1 {
		return nil, errors.New("too many services returned")
	}
	ecsService := servicesState.Services[0]
	return ecsService, nil
}
