package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"os"
	"strings"

	"os/exec"

	"context"

	"os/signal"
	"os/user"
	"time"

	"syscall"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/rollbar"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/cloudwatchlogs"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/ecs"
	"github.com/aws/aws-sdk-go/service/elbv2"
	"github.com/aws/aws-sdk-go/service/iam"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/hashicorp/consul/api"
	"github.com/urfave/cli"
	"golang.org/x/sync/errgroup"
)

type pipelineCommand struct {
	consulHelper *consulHelper
	completes    *completes
	logger       *logger
	awsapi       *awsapi

	in io.Reader
}

func (d *pipelineCommand) subcommands() cli.Command {
	return cli.Command{
		Name:  "pipeline",
		Usage: "Pipeline promotion overrides",
		Subcommands: []cli.Command{
			{
				Name:      "set",
				ArgsUsage: "[team] [service] [environment]",
				Usage:     "Set deployment information for a pipeline.",
				Action:    errExit(d.pipelineSetAction),
				Flags: []cli.Flag{
					cli.StringFlag{
						Name:  "file, f",
						Usage: "File to read pipeline information from.  Defaults to STDIN",
						Value: "-",
					},
				},
			},
			{
				Name:      "credentials",
				ArgsUsage: "[team] [service] [environment]",
				Usage:     "Print out credentials a build should run under",
				Action:    errExit(d.pipelineCredentialsAction),
			},
			{
				Name:      "docker-certify",
				ArgsUsage: "[team] [service] [version]",
				Usage:     "Create a secrets file for jenkins",
				Action:    errExit(d.pipelineDockerCertify),
			},
			{
				Name:      "create-jenkins-secrets",
				ArgsUsage: "[region] [bucket] [key] [profile]",
				Usage:     "Create a secrets file for jenkins",
				Action:    errExit(d.pipelineCreateSecretsFile),
			},
			{
				Name:      "deploy",
				ArgsUsage: "[team] [service] [environment] [tag]",
				Usage:     "Run a full ECS deployment",
				Action:    errExit(d.pipelineDeployAction),
				Flags: []cli.Flag{
					cli.StringFlag{
						Name:  "docker",
						Usage: "Docker host",
						Value: "https://docker.pkgs.xarth.tv",
					},
					cli.BoolFlag{
						Name:  "skiprole, r",
						Usage: "If set, will skip elevating aws roles during deployments.",
					},
				},
			},
			{
				Name:      "create-task",
				ArgsUsage: "[team] [service] [environment] [tag]",
				Usage:     "Create and register task definition, printing it to stdout",
				Action:    errExit(d.pipelineCreateTask),
				Flags: []cli.Flag{
					cli.StringFlag{
						Name:  "docker",
						Usage: "Docker host",
						Value: "https://docker.pkgs.xarth.tv",
					},
					cli.BoolFlag{
						Name:  "skiprole, r",
						Usage: "If set, will skip elevating aws roles during deployments.",
					},
				},
			},
			{
				Name:      "user-deploy",
				ArgsUsage: "[team] [service] [environment] [tag]",
				Usage:     "Let a user created task pretend to be in a service, adding itself to the correct target group.",
				Action:    errExit(d.pipelineUserTask),
				Flags: []cli.Flag{
					cli.StringFlag{
						Name:  "docker",
						Usage: "Docker host",
						Value: "https://docker.pkgs.xarth.tv",
					},
					cli.StringFlag{
						Name:  "user",
						Usage: "Username running this command (put on docker task)",
						Value: "",
					},
					cli.BoolFlag{
						Name:  "skiprole, r",
						Usage: "If set, will skip elevating aws roles during deployments.",
					},
				},
			},
			{
				Name:      "as-json",
				ArgsUsage: "[team] [service] [tag]",
				Usage:     "Print out full pipeline as JSON starting from tag",
				Action:    errExit(d.pipelinePromotionJSON),
			},
			{
				Name:      "dependents",
				ArgsUsage: "[team] [service] [tag]",
				Usage:     "Print to stdout any dependent builds on auto deploy",
				Action:    errExit(d.pipelineDependentsAction),
				Flags: []cli.Flag{
					cli.BoolFlag{
						Name:  "all, a",
						Usage: "If set, print all dependent builds, not just auto deploy ones",
					},
				},
			},
			{
				Name:  "certify",
				Usage: "Certify a SHA as being correctly deployed to an environment",
				Subcommands: []cli.Command{
					{
						Name:         "get",
						ArgsUsage:    "[team] [service] [environment]",
						Usage:        "Get the currently certified SHA for a deployment",
						Action:       errExit(d.certifyGetAction),
						BashComplete: d.completes.teamServiceEnvComplete,
					},
					{
						Name:         "set",
						ArgsUsage:    "[team] [service] [environment] [tag]",
						Usage:        "Set the currently certified SHA for a deployment",
						Action:       errExit(d.certifySetAction),
						BashComplete: d.completes.teamServiceEnvComplete,
					},
				},
			},
			{
				Name:  "environment",
				Usage: "Modify a service environment",
				Subcommands: []cli.Command{
					{
						Name:         "claim",
						ArgsUsage:    "[team] [service] [environment] [tag]",
						Usage:        "Claim an environment as your own, with an overridden tag",
						Action:       errExit(d.pipelineClaimAction),
						BashComplete: d.completes.teamServiceEnvComplete,
					},
					{
						Name:         "release",
						ArgsUsage:    "[team] [service] [environment]",
						Usage:        "Release claim on an environment",
						Action:       errExit(d.pipelineReleaseAction),
						BashComplete: d.completes.teamServiceEnvComplete,
					},
					{
						Name:         "get",
						ArgsUsage:    "[team] [service] [environment]",
						Usage:        "Get the currently overridden pipeline location",
						Action:       errExit(d.pipelineGetAction),
						BashComplete: d.completes.teamServiceEnvComplete,
					},
				},
			},
		},
	}
}

func (d *pipelineCommand) pipelineDependentsAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	tag := c.Args().Get(2)

	// After a tag has been promoted, find every environment under team/service and print out any that are listening
	// to that tag

	envs, err := d.consulHelper.infoServiceEnvs(team, service)
	if err != nil {
		return err
	}

	deps := make([]string, 0, len(envs))
	for _, env := range envs {
		di, err := d.consulHelper.getActualDeploymentInfo(team, service, env)
		if err != nil {
			return err
		}
		if di.PromoteFrom == tag {
			buildName := team + "-" + service + "-" + env
			if c.Bool("all") || di.AutoPromote {
				deps = append(deps, buildName)
			}
		}
	}
	if len(deps) != 0 {
		fmt.Fprintln(c.App.Writer, strings.Join(deps, "\n"))
	}
	return nil
}

type promotionPipeline struct {
	Steps []promotionStep `json:"steps"`
}

type promotionStep struct {
	Auto   []string `json:"auto,omitempty"`
	Manual []string `json:"manual,omitempty"`
}

func (d *pipelineCommand) pipelinePromotionJSON(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	tag := c.Args().Get(2)

	// After a tag has been promoted, find every environment under team/service and print out any that are listening
	// to that tag

	envs, err := d.consulHelper.infoServiceEnvs(team, service)
	if err != nil {
		return err
	}

	stepMap := make(map[string]promotionStep, len(envs))

	for _, env := range envs {
		di, err := d.consulHelper.getDeploymentInfo(team, service, env)
		if err != nil {
			return err
		}
		steps, exists := stepMap[di.PromoteFrom]
		if !exists {
			steps = promotionStep{}
			stepMap[di.PromoteFrom] = steps
		}
		if di.AutoPromote {
			steps.Auto = append(steps.Auto, env)
		} else {
			steps.Manual = append(steps.Auto, env)
		}
		stepMap[di.PromoteFrom] = steps
	}
	ret := promotionPipeline{
		Steps: []promotionStep{},
	}
	promoteAt := []string{tag}
	for len(promoteAt) > 0 {
		currentStep := promoteAt[0]
		promoteAt = promoteAt[1:]
		steps, exists := stepMap[currentStep]
		if !exists {
			continue
		}
		ret.Steps = append(ret.Steps, steps)
		promoteAt = append(promoteAt, steps.Manual...)
		promoteAt = append(promoteAt, steps.Auto...)
	}

	if err := json.NewEncoder(c.App.Writer).Encode(ret); err != nil {
		fmt.Println(c.App.ErrWriter, err)
	}
	return nil
}

func (d *pipelineCommand) pipelineDeployActionSetup(c *cli.Context) (string, string, string, string, *deploymentInfo, error) {
	if c.NArg() != 3 && c.NArg() != 4 {
		return "", "", "", "", nil, cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)
	tag := c.Args().Get(3)
	d.logger.verbose(c, "Deployment started", team, service, env, tag)
	di, err2 := d.consulHelper.getActualDeploymentInfo(team, service, env)
	if err2 != nil {
		return "", "", "", "", nil, err2
	}
	if tag == "" {
		tag = di.PromoteFrom
	}
	return team, service, env, tag, di, nil
}

func (d *pipelineCommand) blockForTaskToRun(ecsClient *ecs.ECS, task *ecs.Task) (*ecs.Task, error) {
	i := 0
	for *task.LastStatus == ecs.DesiredStatusPending && i < 30 {
		fmt.Println("Sleeping till task not in PENDING state...")
		time.Sleep(time.Second)
		taskDesc, err := ecsClient.DescribeTasks(&ecs.DescribeTasksInput{
			Cluster: task.ClusterArn,
			Tasks:   []*string{task.TaskArn},
		})
		if err != nil {
			return nil, err
		}
		if len(taskDesc.Tasks) != 1 {
			return nil, errors.New("could not find task in search")
		}
		task = taskDesc.Tasks[0]
		i++
	}
	if i == 30 {
		return nil, errors.New("task stayed pending")
	}
	return task, nil
}

func (d *pipelineCommand) getCurrentUser(c *cli.Context) string {
	if c.String("user") != "" {
		return c.String("user")
	}
	curUser, err := user.Current()
	if err != nil {
		return "unknown"
	}
	return curUser.Username
}

func (d *pipelineCommand) pipelineUserTask(c *cli.Context) error {
	curUser := d.getCurrentUser(c)
	out, err := d.pipelineCreateTaskAction(c)
	if err != nil {
		return err
	}
	// Task definition created.  Now describe service so we know how to deploy the task
	ecsClient, err := d.awsapi.getECSClient(out.di.Region, out.di.getRole(c), out.di.Profile)
	if err != nil {
		return err
	}
	defer func() {
		fmt.Fprintln(c.App.Writer, "Trying to clean up freshly created task definition")
		_, err2 := ecsClient.DeregisterTaskDefinition(&ecs.DeregisterTaskDefinitionInput{
			TaskDefinition: out.taskDefinition.TaskDefinitionArn,
		})
		if err2 != nil {
			fmt.Fprintf(c.App.ErrWriter, "could not deregister task definition: %s", err2.Error())
		}
		fmt.Fprintln(c.App.Writer, "DeregisterTaskDefinition finished")
	}()
	descOut, err := ecsClient.DescribeServices(&ecs.DescribeServicesInput{
		Cluster:  &out.di.ClusterName,
		Services: []*string{&out.di.ServiceName},
	})
	if err != nil {
		return err
	}
	if len(descOut.Services) != 1 {
		return errors.Errorf("unable to find the service %s", out.di.ServiceName)
	}
	service := descOut.Services[0]
	groupName := fmt.Sprintf("%s-%s-%s", out.team, out.service, curUser)
	taskStartedBy := fmt.Sprintf("ecs-deploy-%s", curUser)
	taskDefinition := fmt.Sprintf("%s:%d", *out.taskDefinition.Family, *out.taskDefinition.Revision)
	fmt.Fprintf(c.App.Writer, "Created task %s\n", taskDefinition)

	runOut, err := ecsClient.RunTask(&ecs.RunTaskInput{
		Cluster:              &out.di.ClusterName,
		Count:                aws.Int64(1),
		Group:                &groupName,
		PlacementConstraints: service.PlacementConstraints,
		PlacementStrategy:    service.PlacementStrategy,
		StartedBy:            &taskStartedBy,
		TaskDefinition:       &taskDefinition,
	})
	if err != nil {
		return err
	}
	if len(runOut.Tasks) != 1 {
		return errors.New("could not find ran task in RunTask output")
	}
	ranTask := runOut.Tasks[0]
	fmt.Printf("Task ran: ARN %s\n", *ranTask.TaskArn)
	fmt.Printf("Waiting for task to run\n")
	var err2 error
	ranTask, err2 = d.blockForTaskToRun(ecsClient, ranTask)
	if err2 != nil {
		return err2
	}

	defer func() {
		fmt.Fprintln(c.App.Writer, "Trying to StopTask")
		_, err2 := ecsClient.StopTask(&ecs.StopTaskInput{
			Cluster: &out.di.ClusterName,
			Reason:  aws.String("ecs-deploy script clean exit"),
			Task:    ranTask.TaskArn,
		})
		if err2 != nil {
			fmt.Fprintf(c.App.ErrWriter, "could not clean stop task: %s", err2.Error())
		}
		fmt.Fprintln(c.App.Writer, "StopTask finished")
	}()

	stdoutStreamStopped := make(chan struct{})
	defer func() {
		fmt.Fprintln(c.App.Writer, "Waiting for stdout stream to stop")
		<-stdoutStreamStopped
		fmt.Fprintln(c.App.Writer, "stdout stream stopped")
	}()

	logClient, err := d.awsapi.getLogClient(out.di.Region, out.di.getRole(c), out.di.Profile)
	if err != nil {
		return err
	}

	ctxToDieOnFinish, cancel := context.WithCancel(context.Background())
	defer cancel()
	go func() {
		if err2 := d.pipelineStreamTaskStdout(ctxToDieOnFinish, c, ranTask, out.taskDefinition, ecsClient, logClient); err2 != nil {
			fmt.Fprintf(c.App.ErrWriter, "unable to finish streaming stdout: %s\n", err2.Error())
		}
		close(stdoutStreamStopped)
	}()

	descContainerOut, err := ecsClient.DescribeContainerInstances(&ecs.DescribeContainerInstancesInput{
		Cluster:            &out.di.ClusterName,
		ContainerInstances: []*string{ranTask.ContainerInstanceArn},
	})
	if err != nil {
		return err
	}
	if len(descContainerOut.ContainerInstances) != 1 {
		return errors.Errorf("too many container instances returned")
	}
	containerInstance := descContainerOut.ContainerInstances[0]

	ec2Client, err := d.awsapi.getEC2Client(out.di.Region, out.di.getRole(c), out.di.Profile)
	if err != nil {
		return err
	}
	descInstancesOut, err := ec2Client.DescribeInstances(&ec2.DescribeInstancesInput{
		InstanceIds: []*string{containerInstance.Ec2InstanceId},
	})
	if err != nil {
		return err
	}
	if len(descInstancesOut.Reservations) != 1 {
		return errors.New("unable to find reservation task is running on")
	}
	if len(descInstancesOut.Reservations[0].Instances) != 1 {
		return errors.New("unable to find instance task is running on")
	}
	instance := descInstancesOut.Reservations[0].Instances[0]
	if instance.PrivateIpAddress == nil {
		return errors.New("unable to find instance task private IP")
	}
	for _, container := range ranTask.Containers {
		fmt.Fprintf(c.App.Writer, "Container: %s\n", *container.Name)
		for _, binding := range container.NetworkBindings {
			fmt.Fprintf(c.App.Writer, "%s %s:%d => %s:%d\n", *binding.Protocol, *binding.BindIP, *binding.ContainerPort, *instance.PrivateIpAddress, *binding.HostPort)
		}
	}

	elbClient, err := d.awsapi.getELBClient(out.di.Region, out.di.getRole(c), out.di.Profile)
	if err != nil {
		return err
	}

	// Update the service target groups to point to the task
	for _, lb := range service.LoadBalancers {
		// find the container for this lb
		for _, container := range ranTask.Containers {
			if *container.Name != *lb.ContainerName {
				continue
			}
			for _, nbind := range container.NetworkBindings {
				if *nbind.ContainerPort != *lb.ContainerPort {
					continue
				}
				fmt.Fprintf(c.App.Writer, "Adding to target group %s value %s:%d\n", *lb.TargetGroupArn, *instance.PrivateIpAddress, *nbind.HostPort)
				_, err := elbClient.RegisterTargets(&elbv2.RegisterTargetsInput{
					TargetGroupArn: lb.TargetGroupArn,
					Targets: []*elbv2.TargetDescription{
						{
							Id:   containerInstance.Ec2InstanceId,
							Port: nbind.HostPort,
						},
					},
				})
				if err != nil {
					return err
				}
				// Yes, an actual use case for defer in a loop doing what I want
				defer func() {
					_, err := elbClient.DeregisterTargets(&elbv2.DeregisterTargetsInput{
						TargetGroupArn: lb.TargetGroupArn,
						Targets: []*elbv2.TargetDescription{
							{
								Id:   containerInstance.Ec2InstanceId,
								Port: nbind.HostPort,
							},
						},
					})
					if err != nil {
						fmt.Fprintf(c.App.ErrWriter, "unable to revert target group modification: %s\n", err.Error())
					}
				}()
				break
			}
			break
		}
	}

	// Trap close signal
	fmt.Fprintln(c.App.Writer, "Waiting for Ctrl+C")
	fmt.Fprintln(c.App.Writer, strings.Repeat("-", 80))
	sigChan := make(chan os.Signal, 2)
	signal.Notify(sigChan, os.Interrupt)
	signal.Notify(sigChan, syscall.SIGTERM)
	<-sigChan
	fmt.Fprintln(c.App.Writer, "Kill signal observed")

	return nil
}

func (p *pipelineCommand) streamSingleContainer(ctx context.Context, c *cli.Context, group string, streamName string, logClient *cloudwatchlogs.CloudWatchLogs) error {
	var prevToken *string
	for {
		p.logger.verbose(c, "getting events from", streamName)
		out, err := logClient.GetLogEventsWithContext(ctx, &cloudwatchlogs.GetLogEventsInput{
			LogGroupName:  &group,
			LogStreamName: &streamName,
			NextToken:     prevToken,
			StartFromHead: aws.Bool(true),
		})
		if err != nil {
			if ctx.Err() == nil {
				return err
			}
			return nil
		}
		p.logger.verbose(c, "got events", len(out.Events))
		prevToken = out.NextForwardToken
		p.logger.verbose(c, "next token", *prevToken)
		if len(out.Events) > 0 {
			for _, event := range out.Events {
				fmt.Println(*event.Message)
			}
			continue
		}
		select {
		case <-ctx.Done():
			p.logger.verbose(c, "stdout kill signal")
			return nil
		case <-time.After(time.Second):
		}
	}
}

func (p *pipelineCommand) pipelineStreamTaskStdout(ctx context.Context, c *cli.Context, task *ecs.Task, taskDef *ecs.TaskDefinition, ecsClient *ecs.ECS, logClient *cloudwatchlogs.CloudWatchLogs) error {
	eg, egCtx := errgroup.WithContext(ctx)
	for _, containerDef := range taskDef.ContainerDefinitions {
		if containerDef.LogConfiguration == nil {
			continue
		}
		if *containerDef.LogConfiguration.LogDriver != "awslogs" {
			continue
		}
		region := keyVal(containerDef.LogConfiguration.Options, "awslogs-region")
		group := keyVal(containerDef.LogConfiguration.Options, "awslogs-group")
		prefix := keyVal(containerDef.LogConfiguration.Options, "awslogs-stream-prefix")
		if region == "" || group == "" {
			return fmt.Errorf("invalid config <%s> <%s>", region, group)
		}
		if region != *ecsClient.Config.Region {
			return fmt.Errorf("unable to get config out of another region: %s", region)
		}
		streamName := fmt.Sprintf("%s/%s/%s", prefix, *containerDef.Name, taskID(*task.TaskArn))
		eg.Go(func() error {
			return p.streamSingleContainer(egCtx, c, group, streamName, logClient)
		})
	}
	return eg.Wait()
}

func (d *pipelineCommand) pipelineCreateTask(c *cli.Context) error {
	out, err := d.pipelineCreateTaskAction(c)
	if err != nil {
		return err
	}
	_, err = fmt.Fprintf(c.App.Writer, "%s:%d\n", *out.taskDefinition.Family, *out.taskDefinition.Revision)
	return err
}

type pipelineCreateTaskOutput struct {
	team           string
	service        string
	env            string
	tag            string
	di             *deploymentInfo
	taskDefinition *ecs.TaskDefinition
}

func (d *pipelineCommand) pipelineCreateTaskAction(c *cli.Context) (*pipelineCreateTaskOutput, error) {
	team, service, env, tag, di, err2 := d.pipelineDeployActionSetup(c)
	if err2 != nil {
		return nil, err2
	}
	if team == "" {
		return nil, errors.New("not enough args")
	}
	d.logger.verbose(c, "getting deployed version for tag", team, service, tag)
	version, err2 := d.consulHelper.deployedVersion(team, service, tag)
	if err2 == nil && version != "" {
		tag = version
	}
	d.logger.verbose(c, "deployed version updated to", tag)

	d.logger.verbose(c, "Deployment info fetched", di)
	d.logger.verbose(c, "Validating docker image")
	if err := dockerImageExists(c, team, service, tag); err != nil {
		return nil, err
	}
	d.logger.verbose(c, "Docker image validated")

	tv := map[string]string{
		"team":           team,
		"service":        service,
		"environment":    env,
		"git_commit":     tag,
		"task_cpu":       "1024",
		"task_mem":       "512",
		"ulimit_nofile":  "10000",
		"container_port": "8080",
		"task_name":      "team" + "-" + service,
		"image":          "docker.pkgs.xarth.tv/" + team + "/" + service + ":" + tag,
	}
	d.logger.verbose(c, "Creating ECS task")
	taskDefinition, err2 := createECSTask(c, di, d.awsapi, d.logger, tv)
	if err2 != nil {
		return nil, err2
	}
	return &pipelineCreateTaskOutput{
		team:           team,
		service:        service,
		env:            env,
		tag:            tag,
		di:             di,
		taskDefinition: taskDefinition,
	}, nil
}

func (d *pipelineCommand) pipelineDeployAction(c *cli.Context) error {
	createOutput, err := d.pipelineCreateTaskAction(c)
	if err != nil {
		return err
	}
	taskDefinition := createOutput.taskDefinition
	di := createOutput.di
	team := createOutput.team
	service := createOutput.service
	env := createOutput.env
	tag := createOutput.tag

	taskFamily := *taskDefinition.Family
	d.logger.verbose(c, "Task created for family", taskFamily)

	d.logger.verbose(c, "Starting deployment", taskFamily)
	if err2 := deployTask(c, taskFamily, d.awsapi, di); err2 != nil {
		return err2
	}

	d.logger.verbose(c, "Finished task deployment", taskFamily)

	d.logger.verbose(c, "Waiting for task to be stable", taskFamily)
	if err2 := blockTillStable(c, d.awsapi, di, taskDefinition); err2 != nil {
		return err2
	}
	d.logger.verbose(c, "Task is stable", taskFamily)

	if err2 := d.setupDeploy(c, team, service, env, tag, di); err2 != nil {
		return err2
	}

	// TODO: Tag docker images

	d.logger.verbose(c, "Cleaning up tasks", taskFamily)
	removed, err := cleanUpTasks(c, d.awsapi, di, taskFamily)
	if err != nil {
		return err
	}
	d.logger.verbose(c, "Task cleanup done", strings.Join(removed, ","))

	return nil
}

func (d *pipelineCommand) setRollbarDeploy(c *cli.Context, team string, service string, environment string, di *deploymentInfo, tag string) error {
	client, err := d.awsapi.getSSMClient(di.Region, di.DeployAWSRole, di.Profile)
	if err != nil {
		return err
	}
	tokenName := fmt.Sprintf("/pipeline/%s/%s/rollbar.deploy.token", team, service)
	res, err := client.GetParameters(&ssm.GetParametersInput{
		WithDecryption: aws.Bool(true),
		Names:          []*string{&tokenName},
	})
	if err != nil {
		if !strings.Contains(err.Error(), "AccessDeniedException") {
			return err
		}
		fmt.Fprintln(c.App.Writer, err)
	}
	if len(res.Parameters) != 1 {
		d.logger.verbose(c, "unable to find rollbar deploy token ", tokenName)
		return nil
	}
	param := res.Parameters[0]
	if param == nil || param.Value == nil {
		return nil
	}
	rollbarClient := rollbar.Client{
		AccessToken: *param.Value,
		Environment: environment,
	}
	return rollbarClient.Deployment(context.Background(), tag, &rollbar.DeploymentOptions{
		Comment: aws.String("ECS promotion deployment.  Build URL: " + os.Getenv("BUILD_URL")),
		// TODO: Find a way to inject the user that started this build into ENV so I can get it out
		LocalUsername: aws.String("ecs-deploy"),
	})
}

func (d *pipelineCommand) setupDeploy(c *cli.Context, team string, service string, env string, tag string, di *deploymentInfo) error {
	d.logger.verbose(c, "certify image as deployed")
	if err := d.consulHelper.setDeployedVersion(team, service, env, tag); err != nil {
		return err
	}
	d.logger.verbose(c, "image certified")

	d.logger.verbose(c, "checking for rollbar token")
	if err := d.setRollbarDeploy(c, team, service, env, di, tag); err != nil {
		// Safe to ignore this error
		d.logger.verbose(c, "unable to set rollbar deployment", err)
	}
	d.logger.verbose(c, "finished checking for rollbar token")
	return nil
}

func (d *pipelineCommand) certifyGetAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)
	version, err := d.consulHelper.deployedVersion(team, service, env)
	if err != nil {
		return err
	}
	fmt.Fprintln(c.App.Writer, version)
	return nil
}

func (d *pipelineCommand) teamSetAction(c *cli.Context) error {
	if c.NArg() != 1 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	kv, err2 := d.consulHelper.createConsulClient()
	if err2 != nil {
		return err2
	}
	var di deploymentInfo
	if err := json.NewDecoder(d.in).Decode(&di); err != nil {
		return err
	}
	encoded, err2 := json.Marshal(di)
	if err2 != nil {
		return err2
	}
	kvp := api.KVPair{
		Key:   "pipeline/info/" + team,
		Value: encoded,
	}
	_, err2 = kv.Put(&kvp, nil)
	if err2 != nil {
		return err2
	}
	fmt.Fprintln(c.App.Writer, "OK")
	return nil
}

type awsCredData struct {
	AccessKey string
	Secret    string
	Profile   string
	RoleARN   string
}

func (d *pipelineCommand) pipelineCreateSecretsFile(c *cli.Context) error {
	if c.NArg() != 4 {
		return cli.ShowSubcommandHelp(c)
	}
	region := c.Args().Get(0)
	bucket := c.Args().Get(1)
	key := c.Args().Get(2)
	profile := c.Args().Get(3)
	tremote := terraformStateRemote{
		Region: region,
		Bucket: bucket,
		Key:    key,
	}
	s3Client, err := d.awsapi.getS3Client(region, "", profile)
	if err != nil {
		return err
	}
	state, err := tremote._readTerraformState(s3Client)
	if err != nil {
		return err
	}
	d.logger.verbose(c, "state loaded", state)
	buildRole, exists := state["build_role"]
	if !exists {
		return errors.New("could not find build_role in s3 state")
	}
	buildUser, exists := state["build_user"]
	if !exists {
		return errors.New("could not find build_user in s3 state")
	}
	iamClient, err := d.awsapi.getIAMClient(region, buildRole, profile)
	if err != nil {
		return err
	}
	userOut, err := iamClient.GetUser(&iam.GetUserInput{
		UserName: &buildUser,
	})
	if err != nil {
		return err
	}
	res, err := iamClient.CreateAccessKey(&iam.CreateAccessKeyInput{
		UserName: userOut.User.UserName,
	})
	if err != nil {
		return err
	}
	data := awsCredData{
		AccessKey: *res.AccessKey.AccessKeyId,
		Secret:    *res.AccessKey.SecretAccessKey,
		RoleARN:   buildRole,
		Profile:   profile,
	}
	t := awsCredsTemplate
	buf := bytes.Buffer{}
	if err := t.Execute(&buf, data); err != nil {
		return err
	}
	fmt.Fprintln(c.App.Writer, buf.String())
	return nil
}

func (d *pipelineCommand) pipelineDockerCertify(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	version := c.Args().Get(2)
	curBranch := os.Getenv("GIT_BRANCH")

	if _, err := exec.LookPath("docker"); err != nil {
		return err
	}

	dockerPath := fmt.Sprintf("%s/%s:%s", team, service, version)
	registry := "docker.pkgs.xarth.tv"

	if err := cmdRun(c, d.logger, "docker", "tag", dockerPath, registry+"/"+dockerPath); err != nil {
		return err
	}

	if err := cmdRun(c, d.logger, "docker", "push", registry+"/"+dockerPath); err != nil {
		return err
	}

	branchSplit := strings.Split(curBranch, "/")
	if len(branchSplit) != 2 {
		return nil
	}
	branchName := branchSplit[1]
	if branchName == "" {
		return nil
	}

	if branchName == "master" {
		branchName = "latest"
	}

	newDockerPath := fmt.Sprintf("%s/%s:%s", team, service, branchName)

	if err := cmdRun(c, d.logger, "docker", "tag", dockerPath, registry+"/"+newDockerPath); err != nil {
		return err
	}

	if err := cmdRun(c, d.logger, "docker", "push", registry+"/"+newDockerPath); err != nil {
		return err
	}

	return d.consulHelper.setDeployedVersion(team, service, branchName, version)
}

func cmdRun(c *cli.Context, logger *logger, cmdName string, args ...string) error {
	logger.verbose(c, cmdName)
	logger.verbose(c, strings.Join(args, " "))
	cmd := exec.Command(cmdName, args...)
	cmd.Stderr = c.App.ErrWriter
	cmd.Stdout = c.App.Writer
	return cmd.Run()
}

func (d *pipelineCommand) pipelineCredentialsAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)
	di, err := d.consulHelper.getActualDeploymentInfo(team, service, env)
	if err != nil {
		return err
	}
	fmt.Fprintln(c.App.Writer, di.AWSCreds)
	return nil
}

func (d *pipelineCommand) pipelineSetAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)

	valueFile := c.String("f")
	fmt.Println(valueFile)
	var readFrom io.Reader
	if valueFile == "-" {
		readFrom = os.Stdin
	} else {
		f, err := os.Open(valueFile)
		if err != nil {
			return err
		}
		defer func() {
			if err := f.Close(); err != nil {
				fmt.Fprintln(c.App.ErrWriter, "unable to close file", err.Error())
			}
		}()
		readFrom = f
	}
	var di deploymentInfo
	if err := json.NewDecoder(readFrom).Decode(&di); err != nil {
		return err
	}
	if err := d.setDeploymentInfo(team, service, env, di); err != nil {
		return err
	}
	fmt.Fprintln(c.App.Writer, "OK")
	return nil
}

func (d *pipelineCommand) setDeploymentInfo(team string, service string, env string, di deploymentInfo) error {
	buf := bytes.Buffer{}
	enc := json.NewEncoder(&buf)
	enc.SetIndent("", "\t")
	err := enc.Encode(di)
	if err != nil {
		return err
	}
	kv, err := d.consulHelper.createConsulClient()
	if err != nil {
		return err
	}
	kvp := api.KVPair{
		Key:   "pipeline/info/" + team + "/" + service + "/" + env,
		Value: buf.Bytes(),
	}
	_, err = kv.Put(&kvp, nil)
	return err
}

func (d *pipelineCommand) teamRemoveAction(c *cli.Context) error {
	if c.NArg() != 1 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	kv, err := d.consulHelper.createConsulClient()
	if err != nil {
		return err
	}
	kvp, _, err := kv.Get("pipeline/teams/"+team, nil)
	if err != nil {
		return err
	}
	if kvp == nil {
		return errors.Errorf("Team not found: %s", team)
	}
	ok, _, err := kv.DeleteCAS(kvp, nil)
	if err != nil {
		return err
	}
	if !ok {
		return errors.Errorf("Race removing team: %s", team)
	}
	fmt.Fprintln(c.App.Writer, "OK")
	return nil
}

func (d *pipelineCommand) teamListAction(c *cli.Context) error {
	if c.NArg() != 0 {
		return cli.ShowSubcommandHelp(c)
	}
	teams, err := d.consulHelper.teams()
	if err != nil {
		return err
	}
	fmt.Fprintln(c.App.Writer, strings.Join(teams, "\n"))
	return nil
}

func (d *pipelineCommand) certifySetAction(c *cli.Context) error {
	if c.NArg() != 4 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)
	tag := c.Args().Get(3)

	if err := d.consulHelper.setDeployedVersion(team, service, env, tag); err != nil {
		return err
	}

	fmt.Fprintln(c.App.Writer, "OK")
	return nil
}

func (d *pipelineCommand) pipelineClaimAction(c *cli.Context) error {
	if c.NArg() != 4 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)
	tag := c.Args().Get(3)

	kv, err := d.consulHelper.createConsulClient()
	if err != nil {
		return err
	}

	kvpair := &api.KVPair{
		Key:   fmt.Sprintf("pipeline/override/%s/%s/%s/promote_from", team, service, env),
		Value: []byte(tag),
	}
	_, err = kv.Put(kvpair, nil)

	return err
}

func (d *pipelineCommand) pipelineReleaseAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)

	kv, err := d.consulHelper.createConsulClient()
	if err != nil {
		return err
	}

	_, err = kv.Delete(fmt.Sprintf("pipeline/override/%s/%s/%s/promote_from", team, service, env), nil)

	return err
}

func (d *pipelineCommand) pipelineGetAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)

	kv, err := d.consulHelper.createConsulClient()
	if err != nil {
		return err
	}

	kvp, _, err := kv.Get(fmt.Sprintf("pipeline/override/%s/%s/%s/promote_from", team, service, env), nil)
	if kvp == nil {
		return nil
	}
	fmt.Fprintln(c.App.Writer, string(kvp.Value))

	return err
}
