#!/bin/bash
set -ex

if [ -z "$TEAM" ]; then
  (>&2 echo "TEAM name not set")
  exit 1
fi

if [ -z "$SERVICE" ]; then
  (>&2 echo "SERVICE name not set")
  exit 1
fi

if [ -z "$ENVIRONMENT" ]; then
  (>&2 echo "ENVIRONMENT name not set")
  exit 1
fi

BUILD_PIPELINE_VERSION="v0.4.7"

rm -f next_step next_desc next_builds

function do_jenkins() {
  echo "Starting jenkins build"
  env
  rm -rf .scratch next_builds next_step next_desc
  mkdir .scratch
  docker pull docker.pkgs.xarth.tv/twitch/ecs-deploy:$BUILD_PIPELINE_VERSION

  echo "read_credentials" > next_step
  echo "read AWS credentials file" > next_desc
}

function do_read_credentials() {
  $(docker_run)/twitch/ecs-deploy:$BUILD_PIPELINE_VERSION /ecs-deploy --verbose pipeline credentials $TEAM $SERVICE $ENVIRONMENT > next_creds
  echo "promotion" > next_step
  echo "Running ECS deployment" > next_desc
}

function do_promotion() {
  $(docker_run)/twitch/ecs-deploy:$BUILD_PIPELINE_VERSION /ecs-deploy --verbose pipeline deploy $TEAM $SERVICE $ENVIRONMENT $GIT_COMMIT
  echo "run_tests" > next_step
  echo "Running post deploy tests" > next_desc
}

function do_run_tests() {
  echo "trigger_deploys" > next_step
  echo "Triggering downstream deploys" > next_desc
}

function do_trigger_deploys() {
  if [ "$SKIP_DOWNSTREAM" == "true" ]; then
    rm -f next_builds
    echo "Skipping downstream builds"
  else
    $(docker_run)/twitch/ecs-deploy:$BUILD_PIPELINE_VERSION /ecs-deploy --verbose pipeline dependents $TEAM $SERVICE $ENVIRONMENT > next_builds
    echo "Next builds are $(cat next_builds)"
  fi
  echo "done" > next_step
}

function docker_run() {
  ADD_CONFIG_ENV=""
  if [ ! -z "$AWS_CONFIG_FILE" ]; then
    ADD_CONFIG_ENV="-e AWS_CONFIG_FILE=/awsconfig -v $AWS_CONFIG_FILE:/awsconfig:ro"
  fi
  echo 'docker run --read-only -u 61000:61000 -e ' "BUILD_URL=$BUILD_URL" ' -e AWS_SDK_LOAD_CONFIG=1 '$ADD_CONFIG_ENV' -v '$(pwd)':/twitch:ro docker.pkgs.xarth.tv'
}

function do_all() {
  echo "This should run the entire pipeline starting at jenkins..."
  exit 1
}

case "$1" in
  jenkins)
    do_jenkins
    ;;
  read_credentials)
    do_read_credentials
    ;;
  promotion)
    do_promotion
    ;;
  run_tests)
    do_run_tests
    ;;
  trigger_deploys)
    do_trigger_deploys
    ;;
  all)
    do_all
    ;;
  *)
  echo "Usage: $0 {...}"
    exit 1
    ;;
esac
